<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\VisitorRegister;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Barryvdh\DomPDF\Facade\Pdf;
use DateTimeZone;

class VisitorRegisterController extends Controller
{
    public function index()
    {
        $visitorRegisters = VisitorRegister::all();
        return response()->json([
            'status' => 200,
            'data' => $visitorRegisters
        ]);
    }

    public function show($id)
    {
        $visitorRegister = VisitorRegister::find($id);
        if ($visitorRegister) {
            return response()->json([
                'status' => 200,
                'data' => $visitorRegister
            ]);
        }
        return response()->json([
            'status' => 404,
            'message' => 'Visitor Register Not Found'
        ], 404);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'SocId' => 'required|exists:society_master,SocId',
            'Vr_RegDate' => 'required|date_format:Y-m-d',
            'Vr_Name' => 'required|string|max:50',
            'Vr_Co_Name' => 'nullable|string|max:50',
            'VT_Id' => 'required|exists:visitor_type,VT_Id',
            'VrU_Id' => 'required|exists:users_master,U_Id',
            'Visit_Mode' => 'required|string|in:P,W',
            'Vr_Purpose' => 'nullable|string|max:255',
            'VR_Vehicle' => 'nullable|string|max:25',
            'Vr_StDate' => 'required|date_format:Y-m-d',
            'Vr_EdDate' => 'nullable|date_format:Y-m-d|after_or_equal:Vr_StDate',
            'Vr_Apr_Stt' => 'required|string|in:P,A,R',
            'Vr_Apr_By' => 'nullable|exists:users_master,U_Id',
            'Vr_Ent_DT' => 'nullable|date_format:H:i:s',
            'Vr_Est_Tm' => 'nullable|date_format:H:i:s',
            'Vr_Ext_DT' => 'nullable|date_format:H:i:s',
            'Vr_Ent_Gt' => 'nullable|exists:gate_master,GM_ID',
            'Vr_Ext_Gt' => 'nullable|exists:gate_master,GM_ID',
            'Vr_Noof_Vst' => 'required|integer|min:1',
            'Vr_In_Pic' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'Vr_Out_Pic' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'Vr_End_Qr' => 'nullable|integer',
            'Vr_Exit_Qr' => 'nullable|integer',
            'OTP' => 'nullable|string|max:10',
            'Vr_Status' => 'required|string|max:1'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 422,
                'errors' => $validator->messages()
            ], 422);
        }

        $data = $request->all();

        // Auto-generate 6-digit OTP
        $data['OTP'] = sprintf("%06d", rand(0, 999999));

        $basePath = public_path('storage');

        // Handle Vr_In_Pic upload
        if ($request->hasFile('Vr_In_Pic')) {
            $image = $request->file('Vr_In_Pic');
            Log::info('Processing Vr_In_Pic:', ['name' => $image->getClientOriginalName()]);
            $originalName = $image->getClientOriginalName();
            $filePath = 'visitor/' . $originalName;
            $destinationPath = $basePath . '/' . $filePath;

            if (file_exists($destinationPath)) {
                $filenameWithoutExt = pathinfo($originalName, PATHINFO_FILENAME);
                $extension = $image->getClientOriginalExtension();
                $newFileName = $filenameWithoutExt . '_' . time() . '.' . $extension;
                $filePath = 'visitor/' . $newFileName;
                $destinationPath = $basePath . '/' . $filePath;
            }

            if (!file_exists(dirname($destinationPath))) {
                Log::info('Creating directory:', ['path' => dirname($destinationPath)]);
                mkdir(dirname($destinationPath), 0755, true);
            }

            Log::info('Moving Vr_In_Pic:', ['from' => $image->getPathname(), 'to' => $destinationPath]);
            $image->move(dirname($destinationPath), basename($filePath));

            $data['Vr_In_Pic'] = url('storage/' . $filePath);
        }

        // Handle Vr_Out_Pic upload
        if ($request->hasFile('Vr_Out_Pic')) {
            $image = $request->file('Vr_Out_Pic');
            Log::info('Processing Vr_Out_Pic:', ['name' => $image->getClientOriginalName()]);
            $originalName = $image->getClientOriginalName();
            $filePath = 'visitor/' . $originalName;
            $destinationPath = $basePath . '/' . $filePath;

            if (file_exists($destinationPath)) {
                $filenameWithoutExt = pathinfo($originalName, PATHINFO_FILENAME);
                $extension = $image->getClientOriginalExtension();
                $newFileName = $filenameWithoutExt . '_' . time() . '.' . $extension;
                $filePath = 'visitor/' . $newFileName;
                $destinationPath = $basePath . '/' . $filePath;
            }

            if (!file_exists(dirname($destinationPath))) {
                Log::info('Creating directory:', ['path' => dirname($destinationPath)]);
                mkdir(dirname($destinationPath), 0755, true);
            }

            Log::info('Moving Vr_Out_Pic:', ['from' => $image->getPathname(), 'to' => $destinationPath]);
            $image->move(dirname($destinationPath), basename($filePath));

            $data['Vr_Out_Pic'] = url('storage/' . $filePath);
        }

        $visitorRegister = VisitorRegister::create($data);

        return response()->json([
            'status' => 201,
            'message' => 'Visitor Register Created Successfully',
            'data' => $visitorRegister
        ], 201);
    }

    public function update(Request $request, $id)
    {
        Log::debug('Update request data:', $request->all());

        $visitorRegister = VisitorRegister::find($id);
        if (!$visitorRegister) {
            return response()->json([
                'status' => 404,
                'message' => 'Visitor Register Not Found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'SocId' => 'sometimes|exists:society_master,SocId',
            'Vr_RegDate' => 'sometimes|date_format:Y-m-d',
            'Vr_Name' => 'sometimes|string|max:50',
            'Vr_Co_Name' => 'sometimes|nullable|string|max:50',
            'VT_Id' => 'sometimes|exists:visitor_type,VT_Id',
            'VrU_Id' => 'sometimes|exists:users_master,U_Id',
            'Visit_Mode' => 'sometimes|string|in:P,W',
            'Vr_Purpose' => 'sometimes|nullable|string|max:255',
            'VR_Vehicle' => 'sometimes|nullable|string|max:25',
            'Vr_StDate' => 'sometimes|date_format:Y-m-d',
            'Vr_EdDate' => 'sometimes|nullable|date_format:Y-m-d|after_or_equal:Vr_StDate',
            'Vr_Apr_Stt' => 'sometimes|string|in:P,A,R',
            'Vr_Apr_By' => 'sometimes|nullable|exists:users_master,U_Id',
            'Vr_Ent_DT' => 'sometimes|nullable|date_format:H:i:s',
            'Vr_Est_Tm' => 'sometimes|nullable|date_format:H:i:s',
            'Vr_Ext_DT' => 'sometimes|nullable|date_format:H:i:s',
            'Vr_Ent_Gt' => 'sometimes|nullable|exists:gate_master,GM_ID',
            'Vr_Ext_Gt' => 'sometimes|nullable|exists:gate_master,GM_ID',
            'Vr_Noof_Vst' => 'sometimes|nullable|integer|min:1',
            'Vr_In_Pic' => [
                'sometimes',
                'nullable',
                'string',
                function ($attribute, $value, $fail) {
                    if (preg_match('/^https?:\/\/.*\/storage\/visitor\/(.+\.(jpg|jpeg|png))$/i', $value)) {
                        Log::debug("Valid URL detected for $attribute", ['value' => substr($value, 0, 100)]);
                        return;
                    }
                    if (preg_match('/^data:image\/(jpeg|png|jpg);base64,(.+)/', $value, $matches)) {
                        $base64Data = $matches[2];
                        $decodedData = base64_decode($base64Data, true);
                        if ($decodedData === false) {
                            Log::error("Base64 decoding failed for $attribute", ['value' => substr($value, 0, 100)]);
                            $fail("The $attribute contains invalid Base64 data.");
                            return;
                        }
                        if (strlen($decodedData) > 20480 * 1024) {
                            Log::error("File size too large for $attribute", ['size' => strlen($decodedData)]);
                            $fail("The $attribute file size must not exceed 20480 KB.");
                        }
                        Log::debug("Valid Base64 data with data: prefix for $attribute", ['size' => strlen($decodedData)]);
                        return;
                    }
                    if (preg_match('/^[A-Za-z0-9+\/=]+$/', $value)) {
                        $decodedData = base64_decode($value, true);
                        if ($decodedData === false) {
                            Log::error("Base64 decoding failed for raw Base64 $attribute");
                            $fail("The $attribute contains invalid Base64 data.");
                            return;
                        }
                        if (strlen($decodedData) > 20480 * 1024) {
                            Log::error("File size too large for raw Base64 $attribute", ['size' => strlen($decodedData)]);
                            $fail("The $attribute file size must not exceed 20480 KB.");
                        }
                        Log::debug("Valid raw Base64 data for $attribute", ['size' => strlen($decodedData)]);
                        return;
                    }
                    Log::error("Invalid format for $attribute", ['value' => substr($value, 0, 100)]);
                    $fail("The $attribute must be a valid Base64 string of type: jpg, jpeg, png, or a valid file URL.");
                },
            ],
            'Vr_Out_Pic' => [
                'sometimes',
                'nullable',
                'string',
                function ($attribute, $value, $fail) {
                    if (preg_match('/^https?:\/\/.*\/storage\/visitor\/(.+\.(jpg|jpeg|png))$/i', $value)) {
                        Log::debug("Valid URL detected for $attribute", ['value' => substr($value, 0, 100)]);
                        return;
                    }
                    if (preg_match('/^data:image\/(jpeg|png|jpg);base64,(.+)/', $value, $matches)) {
                        $base64Data = $matches[2];
                        $decodedData = base64_decode($base64Data, true);
                        if ($decodedData === false) {
                            Log::error("Base64 decoding failed for $attribute", ['value' => substr($value, 0, 100)]);
                            $fail("The $attribute contains invalid Base64 data.");
                            return;
                        }
                        if (strlen($decodedData) > 20480 * 1024) {
                            Log::error("File size too large for $attribute", ['size' => strlen($decodedData)]);
                            $fail("The $attribute file size must not exceed 20480 KB.");
                        }
                        Log::debug("Valid Base64 data with data: prefix for $attribute", ['size' => strlen($decodedData)]);
                        return;
                    }
                    if (preg_match('/^[A-Za-z0-9+\/=]+$/', $value)) {
                        $decodedData = base64_decode($value, true);
                        if ($decodedData === false) {
                            Log::error("Base64 decoding failed for raw Base64 $attribute");
                            $fail("The $attribute contains invalid Base64 data.");
                            return;
                        }
                        if (strlen($decodedData) > 20480 * 1024) {
                            Log::error("File size too large for raw Base64 $attribute", ['size' => strlen($decodedData)]);
                            $fail("The $attribute file size must not exceed 20480 KB.");
                        }
                        Log::debug("Valid raw Base64 data for $attribute", ['size' => strlen($decodedData)]);
                        return;
                    }
                    Log::error("Invalid format for $attribute", ['value' => substr($value, 0, 100)]);
                    $fail("The $attribute must be a valid Base64 string of type: jpg, jpeg, png, or a valid file URL.");
                },
            ],
            'Vr_End_Qr' => 'sometimes|nullable|integer',
            'Vr_Exit_Qr' => 'sometimes|nullable|integer',
            'OTP' => 'sometimes|nullable|string|max:10',
            'Vr_Status' => 'sometimes|string|max:1'
        ]);

        if ($validator->fails()) {
            Log::error('Validation failed:', ['errors' => $validator->messages()->toArray()]);
            return response()->json([
                'status' => 422,
                'errors' => $validator->messages()
            ], 422);
        }

        $data = $request->only([
            'SocId',
            'Vr_RegDate',
            'Vr_Name',
            'Vr_Co_Name',
            'VT_Id',
            'VrU_Id',
            'Visit_Mode',
            'Vr_Purpose',
            'VR_Vehicle',
            'Vr_StDate',
            'Vr_EdDate',
            'Vr_Apr_Stt',
            'Vr_Apr_By',
            'Vr_Ent_DT',
            'Vr_Est_Tm',
            'Vr_Ext_DT',
            'Vr_Ent_Gt',
            'Vr_Ext_Gt',
            'Vr_Noof_Vst',
            'Vr_In_Pic',
            'Vr_Out_Pic',
            'Vr_End_Qr',
            'Vr_Exit_Qr',
            'OTP',
            'Vr_Status'
        ]);

        $basePath = public_path('storage');

        $detectFileExtension = function ($decodedData) {
            $signatures = [
                "\xFF\xD8\xFF" => 'jpg',
                "\x89\x50\x4E\x47\x0D\x0A\x1A\x0A" => 'png',
            ];
            foreach ($signatures as $signature => $ext) {
                if (strpos($decodedData, $signature) === 0) {
                    return $ext;
                }
            }
            return 'jpg';
        };

        $getExtensionFromMimeType = function ($mimeType) {
            $mimeMap = [
                'image/jpeg' => 'jpg',
                'image/png' => 'png',
                'image/jpg' => 'jpg',
            ];
            return $mimeMap[$mimeType] ?? 'jpg';
        };

        $processImage = function ($item, $fieldName, $id, $prefix) use ($basePath, $detectFileExtension, $getExtensionFromMimeType) {
            if (preg_match('/^https?:\/\/.*\/storage\/visitor\/(.+\.(jpg|jpeg|png))$/i', $item, $matches)) {
                $relativePath = 'visitor/' . $matches[1];
                Log::info("Preserved existing file URL for {$fieldName}", ['path' => $relativePath]);
                return ['url' => url('storage/' . $relativePath), 'status' => 200];
            }

            $base64Data = $item;
            $extension = null;
            $mimeType = null;

            if (preg_match('/^data:image\/(jpeg|png|jpg);base64,(.+)/', $item, $matches)) {
                $mimeType = $matches[1];
                $base64Data = $matches[2];
                $extension = $getExtensionFromMimeType("image/{$mimeType}");
            } else {
                $decodedData = base64_decode($item, true);
                if ($decodedData === false) {
                    Log::error("Base64 decoding failed for {$fieldName}", ['item' => substr($item, 0, 100)]);
                    return ['error' => ["Invalid Base64 data for {$fieldName}"], 'status' => 422];
                }
                $extension = $detectFileExtension($decodedData);
                $base64Data = $item;
            }

            $fileData = base64_decode($base64Data, true);
            if ($fileData === false) {
                Log::error("Base64 decoding failed for {$fieldName}", ['item' => substr($item, 0, 100)]);
                return ['error' => ["Invalid Base64 data for {$fieldName}"], 'status' => 422];
            }

            $fileName = "{$prefix}_{$id}_" . time() . '.' . $extension;
            $filePath = "visitor/{$fileName}";
            $destinationPath = "{$basePath}/{$filePath}";

            if (!file_exists(dirname($destinationPath))) {
                Log::info("Creating directory for {$fieldName}:", ['path' => dirname($destinationPath)]);
                mkdir(dirname($destinationPath), 0755, true);
            }

            $bytesWritten = file_put_contents($destinationPath, $fileData);
            if ($bytesWritten === false) {
                Log::error("Failed to save file for {$fieldName}", ['path' => $filePath]);
                return ['error' => ["Failed to save {$fieldName} image"], 'status' => 500];
            }
            Log::info("Saved file for {$fieldName}", ['path' => $filePath, 'size' => strlen($fileData)]);

            return ['url' => url("storage/{$filePath}"), 'status' => 200];
        };

        if ($request->has('Vr_In_Pic') && !empty($request->Vr_In_Pic)) {
            $result = $processImage($request->Vr_In_Pic, 'Vr_In_Pic', $id, 'visitor_in');
            if (isset($result['error'])) {
                return response()->json([
                    'status' => $result['status'],
                    'errors' => [$result['error']]
                ], $result['status']);
            }

            $data['Vr_In_Pic'] = $result['url'];
            Log::info('Vr_In_Pic URL updated:', ['url' => $data['Vr_In_Pic']]);

            if ($visitorRegister->Vr_In_Pic) {
                $oldPath = str_replace(url('/storage'), public_path('storage'), $visitorRegister->Vr_In_Pic);
                if (file_exists($oldPath)) {
                    Log::info('Deleting old Vr_In_Pic:', ['path' => $oldPath]);
                    unlink($oldPath);
                }
            }
        }

        if ($request->has('Vr_Out_Pic') && !empty($request->Vr_Out_Pic)) {
            $result = $processImage($request->Vr_Out_Pic, 'Vr_Out_Pic', $id, 'visitor_out');
            if (isset($result['error'])) {
                return response()->json([
                    'status' => $result['status'],
                    'errors' => [$result['error']]
                ], $result['status']);
            }

            $data['Vr_Out_Pic'] = $result['url'];
            Log::info('Vr_Out_Pic URL updated:', ['url' => $data['Vr_Out_Pic']]);

            if ($visitorRegister->Vr_Out_Pic) {
                $oldPath = str_replace(url('/storage'), public_path('storage'), $visitorRegister->Vr_Out_Pic);
                if (file_exists($oldPath)) {
                    Log::info('Deleting old Vr_Out_Pic:', ['path' => $oldPath]);
                    unlink($oldPath);
                }
            }
        }

        try {
            $visitorRegister->update($data);
            Log::info('Visitor Register updated successfully:', ['id' => $id, 'data' => $data]);
        } catch (\Exception $e) {
            Log::error('Failed to update Visitor Register:', ['id' => $id, 'error' => $e->getMessage()]);
            return response()->json([
                'status' => 500,
                'message' => 'Failed to update Visitor Register',
                'error' => $e->getMessage()
            ], 500);
        }

        return response()->json([
            'status' => 200,
            'message' => 'Visitor Register Updated Successfully',
            'data' => $visitorRegister->fresh()
        ]);
    }

    public function destroy($id)
    {
        $visitorRegister = VisitorRegister::find($id);
        if ($visitorRegister) {
            $visitorRegister->delete();
            return response()->json([
                'status' => 200,
                'message' => 'Visitor Register Deleted Successfully'
            ]);
        }
        return response()->json([
            'status' => 404,
            'message' => 'Visitor Register Not Found'
        ], 404);
    }

    /**
     * Filter VisitorRegister records by all fields.
     */
    public function filterVisitors(Request $request)
    {
        // Validate filter parameters
        $validator = Validator::make($request->all(), [
            'VR_Id' => 'nullable|integer|exists:visitor_register,VR_Id',
            'SocId' => 'nullable|integer|exists:society_master,SocId',
            'Vr_RegDate' => 'nullable|date_format:Y-m-d',
            'Vr_Name' => 'nullable|string|max:50',
            'Vr_Co_Name' => 'nullable|string|max:50',
            'VT_Id' => 'nullable|integer|exists:visitor_type,VT_Id',
            'VrU_Id' => 'nullable|integer|exists:users_master,U_Id',
            'Visit_Mode' => 'nullable|string|in:P,W',
            'Vr_Purpose' => 'nullable|string|max:255',
            'VR_Vehicle' => 'nullable|string|max:25',
            'Vr_StDate' => 'nullable|date_format:Y-m-d',
            'Vr_EdDate' => 'nullable|date_format:Y-m-d|after_or_equal:Vr_StDate',
            'Vr_Apr_Stt' => 'nullable|string|in:P,A,R',
            'Vr_Apr_By' => 'nullable|integer|exists:users_master,U_Id',
            'Vr_Ent_DT' => 'nullable|date_format:H:i:s',
            'Vr_Est_Tm' => 'nullable|date_format:H:i:s',
            'Vr_Ext_DT' => 'nullable|date_format:H:i:s',
            'Vr_Ent_Gt' => 'nullable|integer|exists:gate_master,GM_ID',
            'Vr_Ext_Gt' => 'nullable|integer|exists:gate_master,GM_ID',
            'Vr_Noof_Vst' => 'nullable|integer|min:1',
            'Vr_In_Pic' => 'nullable|string',
            'Vr_Out_Pic' => 'nullable|string',
            'Vr_End_Qr' => 'nullable|integer',
            'Vr_Exit_Qr' => 'nullable|integer',
            'OTP' => 'nullable|string|max:10',
            'Vr_Status' => 'nullable|string|max:1',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 422,
                'errors' => $validator->messages()
            ], 422);
        }

        // Build query
        $query = VisitorRegister::with(['society', 'visitorType', 'user', 'approvedBy', 'entryGate', 'exitGate']);

        if ($request->has('VR_Id')) {
            $query->where('VR_Id', $request->input('VR_Id'));
        }

        if ($request->has('SocId')) {
            $query->where('SocId', $request->input('SocId'));
        }

        if ($request->has('Vr_RegDate')) {
            $query->whereDate('Vr_RegDate', $request->input('Vr_RegDate'));
        }

        if ($request->has('Vr_Name')) {
            $query->where('Vr_Name', 'like', '%' . $request->input('Vr_Name') . '%');
        }

        if ($request->has('Vr_Co_Name')) {
            $query->where('Vr_Co_Name', 'like', '%' . $request->input('Vr_Co_Name') . '%');
        }

        if ($request->has('VT_Id')) {
            $query->where('VT_Id', $request->input('VT_Id'));
        }

        if ($request->has('VrU_Id')) {
            $query->where('VrU_Id', $request->input('VrU_Id'));
        }

        if ($request->has('Visit_Mode')) {
            $query->where('Visit_Mode', $request->input('Visit_Mode'));
        }

        if ($request->has('Vr_Purpose')) {
            $query->where('Vr_Purpose', 'like', '%' . $request->input('Vr_Purpose') . '%');
        }

        if ($request->has('VR_Vehicle')) {
            $query->where('VR_Vehicle', 'like', '%' . $request->input('VR_Vehicle') . '%');
        }

        if ($request->has('Vr_StDate')) {
            $query->whereDate('Vr_StDate', $request->input('Vr_StDate'));
        }

        if ($request->has('Vr_EdDate')) {
            $query->whereDate('Vr_EdDate', $request->input('Vr_EdDate'));
        }

        if ($request->has('Vr_Apr_Stt')) {
            $query->where('Vr_Apr_Stt', $request->input('Vr_Apr_Stt'));
        }

        if ($request->has('Vr_Apr_By')) {
            $query->where('Vr_Apr_By', $request->input('Vr_Apr_By'));
        }

        if ($request->has('Vr_Ent_DT')) {
            $query->whereTime('Vr_Ent_DT', $request->input('Vr_Ent_DT'));
        }

        if ($request->has('Vr_Est_Tm')) {
            $query->whereTime('Vr_Est_Tm', $request->input('Vr_Est_Tm'));
        }

        if ($request->has('Vr_Ext_DT')) {
            $query->whereTime('Vr_Ext_DT', $request->input('Vr_Ext_DT'));
        }

        if ($request->has('Vr_Ent_Gt')) {
            $query->where('Vr_Ent_Gt', $request->input('Vr_Ent_Gt'));
        }

        if ($request->has('Vr_Ext_Gt')) {
            $query->where('Vr_Ext_Gt', $request->input('Vr_Ext_Gt'));
        }

        if ($request->has('Vr_Noof_Vst')) {
            $query->where('Vr_Noof_Vst', $request->input('Vr_Noof_Vst'));
        }

        if ($request->has('Vr_In_Pic')) {
            $query->where('Vr_In_Pic', 'like', '%' . $request->input('Vr_In_Pic') . '%');
        }

        if ($request->has('Vr_Out_Pic')) {
            $query->where('Vr_Out_Pic', 'like', '%' . $request->input('Vr_Out_Pic') . '%');
        }

        if ($request->has('Vr_End_Qr')) {
            $query->where('Vr_End_Qr', $request->input('Vr_End_Qr'));
        }

        if ($request->has('Vr_Exit_Qr')) {
            $query->where('Vr_Exit_Qr', $request->input('Vr_Exit_Qr'));
        }

        if ($request->has('OTP')) {
            $query->where('OTP', $request->input('OTP'));
        }

        if ($request->has('Vr_Status')) {
            $query->where('Vr_Status', $request->input('Vr_Status'));
        }

        $visitors = $query->get();

        return response()->json([
            'status' => 200,
            'data' => $visitors
        ]);
    }

    /**
     * Generate PDF for filtered VisitorRegister records.
     */
    public function generateVisitorPDF(Request $request)
    {
        // Validate filter parameters
        $validator = Validator::make($request->all(), [
            'VR_Id' => 'nullable|integer|exists:visitor_register,VR_Id',
            'SocId' => 'nullable|integer|exists:society_master,SocId',
            'Vr_RegDate' => 'nullable|date_format:Y-m-d',
            'Vr_Name' => 'nullable|string|max:50',
            'Vr_Co_Name' => 'nullable|string|max:50',
            'VT_Id' => 'nullable|integer|exists:visitor_type,VT_Id',
            'VrU_Id' => 'nullable|integer|exists:users_master,U_Id',
            'Visit_Mode' => 'nullable|string|in:P,W',
            'Vr_Purpose' => 'nullable|string|max:255',
            'VR_Vehicle' => 'nullable|string|max:25',
            'Vr_StDate' => 'nullable|date_format:Y-m-d',
            'Vr_EdDate' => 'nullable|date_format:Y-m-d|after_or_equal:Vr_StDate',
            'Vr_Apr_Stt' => 'nullable|string|in:P,A,R',
            'Vr_Apr_By' => 'nullable|integer|exists:users_master,U_Id',
            'Vr_Ent_DT' => 'nullable|date_format:H:i:s',
            'Vr_Est_Tm' => 'nullable|date_format:H:i:s',
            'Vr_Ext_DT' => 'nullable|date_format:H:i:s',
            'Vr_Ent_Gt' => 'nullable|integer|exists:gate_master,GM_ID',
            'Vr_Ext_Gt' => 'nullable|integer|exists:gate_master,GM_ID',
            'Vr_Noof_Vst' => 'nullable|integer|min:1',
            'Vr_In_Pic' => 'nullable|string',
            'Vr_Out_Pic' => 'nullable|string',
            'Vr_End_Qr' => 'nullable|integer',
            'Vr_Exit_Qr' => 'nullable|integer',
            'OTP' => 'nullable|string|max:10',
            'Vr_Status' => 'nullable|string|max:1',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 422,
                'errors' => $validator->messages()
            ], 422);
        }

        // Build query
        $query = VisitorRegister::with(['society', 'visitorType', 'user', 'approvedBy', 'entryGate', 'exitGate']);

        if ($request->has('VR_Id')) {
            $query->where('VR_Id', $request->input('VR_Id'));
        }

        if ($request->has('SocId')) {
            $query->where('SocId', $request->input('SocId'));
        }

        if ($request->has('Vr_RegDate')) {
            $query->whereDate('Vr_RegDate', $request->input('Vr_RegDate'));
        }

        if ($request->has('Vr_Name')) {
            $query->where('Vr_Name', 'like', '%' . $request->input('Vr_Name') . '%');
        }

        if ($request->has('Vr_Co_Name')) {
            $query->where('Vr_Co_Name', 'like', '%' . $request->input('Vr_Co_Name') . '%');
        }

        if ($request->has('VT_Id')) {
            $query->where('VT_Id', $request->input('VT_Id'));
        }

        if ($request->has('VrU_Id')) {
            $query->where('VrU_Id', $request->input('VrU_Id'));
        }

        if ($request->has('Visit_Mode')) {
            $query->where('Visit_Mode', $request->input('Visit_Mode'));
        }

        if ($request->has('Vr_Purpose')) {
            $query->where('Vr_Purpose', 'like', '%' . $request->input('Vr_Purpose') . '%');
        }

        if ($request->has('VR_Vehicle')) {
            $query->where('VR_Vehicle', 'like', '%' . $request->input('VR_Vehicle') . '%');
        }

        if ($request->has('Vr_StDate')) {
            $query->whereDate('Vr_StDate', $request->input('Vr_StDate'));
        }

        if ($request->has('Vr_EdDate')) {
            $query->whereDate('Vr_EdDate', $request->input('Vr_EdDate'));
        }

        if ($request->has('Vr_Apr_Stt')) {
            $query->where('Vr_Apr_Stt', $request->input('Vr_Apr_Stt'));
        }

        if ($request->has('Vr_Apr_By')) {
            $query->where('Vr_Apr_By', $request->input('Vr_Apr_By'));
        }

        if ($request->has('Vr_Ent_DT')) {
            $query->whereTime('Vr_Ent_DT', $request->input('Vr_Ent_DT'));
        }

        if ($request->has('Vr_Est_Tm')) {
            $query->whereTime('Vr_Est_Tm', $request->input('Vr_Est_Tm'));
        }

        if ($request->has('Vr_Ext_DT')) {
            $query->whereTime('Vr_Ext_DT', $request->input('Vr_Ext_DT'));
        }

        if ($request->has('Vr_Ent_Gt')) {
            $query->where('Vr_Ent_Gt', $request->input('Vr_Ent_Gt'));
        }

        if ($request->has('Vr_Ext_Gt')) {
            $query->where('Vr_Ext_Gt', $request->input('Vr_Ext_Gt'));
        }

        if ($request->has('Vr_Noof_Vst')) {
            $query->where('Vr_Noof_Vst', $request->input('Vr_Noof_Vst'));
        }

        if ($request->has('Vr_In_Pic')) {
            $query->where('Vr_In_Pic', 'like', '%' . $request->input('Vr_In_Pic') . '%');
        }

        if ($request->has('Vr_Out_Pic')) {
            $query->where('Vr_Out_Pic', 'like', '%' . $request->input('Vr_Out_Pic') . '%');
        }

        if ($request->has('Vr_End_Qr')) {
            $query->where('Vr_End_Qr', $request->input('Vr_End_Qr'));
        }

        if ($request->has('Vr_Exit_Qr')) {
            $query->where('Vr_Exit_Qr', $request->input('Vr_Exit_Qr'));
        }

        if ($request->has('OTP')) {
            $query->where('OTP', $request->input('OTP'));
        }

        if ($request->has('Vr_Status')) {
            $query->where('Vr_Status', $request->input('Vr_Status'));
        }

        $visitors = $query->get();

        // Generate PDF
        $pdf = Pdf::loadView('pdf.visitor', compact('visitors'));
        return $pdf->download('visitor_report.pdf');
    }


    public function filterVisitorEntryLogs(Request $request)
    {
        // Set timezone to IST
        date_default_timezone_set('Asia/Kolkata');

        // Validate filter parameters
        $validator = Validator::make($request->all(), [
            'VR_Id' => 'nullable|integer|exists:visitor_registers,VR_Id',
            'SocId' => 'nullable|integer|exists:society_master,SocId',
            'Vr_Name' => 'nullable|string|max:100',
            'Vr_Ent_DT' => 'nullable|date_format:Y-m-d H:i:s',
            'Visit_Mode' => 'nullable|string|max:1',
            'Vr_Status' => 'nullable|string|max:1',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // Build query
        $query = VisitorRegister::with(['society', 'user', 'visitorType']);

        if ($request->has('VR_Id')) {
            $query->where('VR_Id', $request->input('VR_Id'));
        }

        if ($request->has('SocId')) {
            $query->where('SocId', $request->input('SocId'));
        }

        if ($request->has('Vr_Name')) {
            $query->where('Vr_Name', 'like', '%' . $request->input('Vr_Name') . '%');
        }

        if ($request->has('Vr_Ent_DT')) {
            $query->where('Vr_Ent_DT', $request->input('Vr_Ent_DT'));
        }

        if ($request->has('Visit_Mode')) {
            $query->where('Visit_Mode', $request->input('Visit_Mode'));
        }

        if ($request->has('Vr_Status')) {
            $query->where('Vr_Status', $request->input('Vr_Status'));
        }

        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('Vr_Ent_DT', [$request->input('start_date'), $request->input('end_date')]);
        }

        $visitors = $query->get();

        return response()->json($visitors);
    }

    /**
     * Generate PDF for filtered VisitorRegister entry logs.
     */
    public function generateVisitorEntryPDF(Request $request)
    {
        // Set timezone to IST
        date_default_timezone_set('Asia/Kolkata');

        // Validate filter parameters
        $validator = Validator::make($request->all(), [
            'VR_Id' => 'nullable|integer|exists:visitor_registers,VR_Id',
            'SocId' => 'nullable|integer|exists:society_master,SocId',
            'Vr_Name' => 'nullable|string|max:100',
            'Vr_Ent_DT' => 'nullable|date_format:Y-m-d H:i:s',
            'Visit_Mode' => 'nullable|string|max:1',
            'Vr_Status' => 'nullable|string|max:1',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // Build query
        $query = VisitorRegister::with(['society', 'user', 'visitorType']);

        if ($request->has('VR_Id')) {
            $query->where('VR_Id', $request->input('VR_Id'));
        }

        if ($request->has('SocId')) {
            $query->where('SocId', $request->input('SocId'));
        }

        if ($request->has('Vr_Name')) {
            $query->where('Vr_Name', 'like', '%' . $request->input('Vr_Name') . '%');
        }

        if ($request->has('Vr_Ent_DT')) {
            $query->where('Vr_Ent_DT', $request->input('Vr_Ent_DT'));
        }

        if ($request->has('Visit_Mode')) {
            $query->where('Visit_Mode', $request->input('Visit_Mode'));
        }

        if ($request->has('Vr_Status')) {
            $query->where('Vr_Status', $request->input('Vr_Status'));
        }

        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('Vr_Ent_DT', [$request->input('start_date'), $request->input('end_date')]);
        }

        $visitors = $query->get();
        $socId = $request->input('SocId');
        $socName = $socId ? $visitors->first()->society->SocName ?? 'All Societies' : 'All Societies';

        // Generate PDF
        $pdf = Pdf::loadView('pdf.visitor_entry', compact('visitors', 'socName'));
        return $pdf->download('visitor_entry_report.pdf');
    }
}
