<?php

namespace App\Http\Controllers;

use App\Models\VendorRegister;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class VendorRegisterController extends Controller
{
    /**
     * Get all VendorRegister records.
     */
    public function index()
    {
        $vendors = VendorRegister::all();
        return response()->json($vendors);
    }

    /**
     * Get a specific VendorRegister record by Vd_Id.
     */
    public function show($id)
    {
        $vendor = VendorRegister::find($id);

        if (!$vendor) {
            return response()->json(['message' => 'Vendor not found'], 404);
        }

        return response()->json($vendor);
    }

    /**
     * Create a new VendorRegister record.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'BusinessName' => 'nullable|string|max:255',
            'Registration' => 'nullable|string|max:25',
            'TaxRegistration' => 'nullable|string|max:25',
            'STDate' => 'nullable|date_format:d/m/Y',
            'EdDate' => 'nullable|date_format:d/m/Y|after_or_equal:STDate',
            'Status' => 'nullable|string|in:A,I|max:1',
            'VendorType' => 'nullable|string|in:C,V,P|max:1',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $validated = $validator->validated();

        // Convert date fields to MySQL format
        $validated['STDate'] = $request->filled('STDate')
            ? Carbon::createFromFormat('d/m/Y', $request->input('STDate'), 'Asia/Kolkata')
                ->format('Y-m-d')
            : null;
        $validated['EdDate'] = $request->filled('EdDate')
            ? Carbon::createFromFormat('d/m/Y', $request->input('EdDate'), 'Asia/Kolkata')
                ->format('Y-m-d')
            : null;

        $vendor = VendorRegister::create($validated);

        return response()->json($vendor, 201);
    }

    /**
     * Update a VendorRegister record by Vd_Id.
     */
    public function update(Request $request, $id)
    {
        $vendor = VendorRegister::find($id);

        if (!$vendor) {
            return response()->json(['message' => 'Vendor not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'BusinessName' => 'nullable|string|max:255',
            'Registration' => 'nullable|string|max:25',
            'TaxRegistration' => 'nullable|string|max:25',
            'STDate' => 'nullable|date_format:d/m/Y',
            'EdDate' => 'nullable|date_format:d/m/Y|after_or_equal:STDate',
            'Status' => 'nullable|string|in:A,I|max:1',
            'VendorType' => 'nullable|string|in:C,V,P|max:1',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $validated = array_filter($validator->validated());

        // Convert date fields to MySQL format
        if ($request->filled('STDate')) {
            $validated['STDate'] = Carbon::createFromFormat('d/m/Y', $request->input('STDate'), 'Asia/Kolkata')
                ->format('Y-m-d');
        }
        if ($request->filled('EdDate')) {
            $validated['EdDate'] = Carbon::createFromFormat('d/m/Y', $request->input('EdDate'), 'Asia/Kolkata')
                ->format('Y-m-d');
        }

        $vendor->update($validated);

        return response()->json($vendor);
    }

    /**
     * Delete a VendorRegister record by Vd_Id.
     */
    public function destroy($id)
    {
        $vendor = VendorRegister::find($id);

        if (!$vendor) {
            return response()->json(['message' => 'Vendor not found'], 404);
        }

        $vendor->delete();
        return response()->json(['message' => 'Vendor deleted successfully']);
    }
}
