<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\VehicleMaster;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class VehicleMasterController extends Controller
{
    public function index()
    {
        $vehicles = VehicleMaster::with(['unit.block.division', 'society', 'parkingMaster'])->get()->map(function ($vehicle) {
            if ($vehicle->Vehicle_Img) {
                $imageUrls = is_string($vehicle->Vehicle_Img) ? json_decode($vehicle->Vehicle_Img, true) : $vehicle->Vehicle_Img;
                if (is_array($imageUrls)) {
                    $vehicle->Vehicle_Img = array_map(function ($url) {
                        return str_replace('\/', '/', $url);
                    }, $imageUrls);
                } else {
                    $vehicle->Vehicle_Img = [];
                }
            }
            // Convert ActiveDate to IST for response
            if ($vehicle->ActiveDate) {
                $vehicle->ActiveDate = Carbon::parse($vehicle->ActiveDate)->setTimezone('Asia/Kolkata')->format('Y-m-d H:i:s');
            }
            return $vehicle;
        });

        return response()->json($vehicles);
    }

    public function show($id)
    {
        $vehicle = VehicleMaster::with(['unit.block.division', 'society', 'parkingMaster'])->find($id);
        if (!$vehicle) {
            return response()->json(['message' => 'Vehicle not found'], 404);
        }

        if ($vehicle->Vehicle_Img) {
            $imageUrls = is_string($vehicle->Vehicle_Img) ? json_decode($vehicle->Vehicle_Img, true) : $vehicle->Vehicle_Img;
            if (is_array($imageUrls)) {
                $vehicle->Vehicle_Img = array_map(function ($url) {
                    return str_replace('\/', '/', $url);
                }, $imageUrls);
            } else {
                $vehicle->Vehicle_Img = [];
            }
        }
        // Convert ActiveDate to IST for response
        if ($vehicle->ActiveDate) {
            $vehicle->ActiveDate = Carbon::parse($vehicle->ActiveDate)->setTimezone('Asia/Kolkata')->format('Y-m-d H:i:s');
        }

        return response()->json($vehicle);
    }

    public function store(Request $request)
    {
        try {
            Log::info('Store request received:', $request->all());
            // Log the number of files instead of array_keys to avoid issues
            Log::info('Files count:', ['Vehicle_Img' => $request->hasFile('Vehicle_Img') ? count($request->file('Vehicle_Img')) : 0]);

            // Prepare data for validation, excluding Vehicle_Img to avoid string conversion issues
            $data = $request->except('Vehicle_Img');
            // Add only valid files for validation
            $files = $request->file('Vehicle_Img', []);
            foreach ($files as $index => $file) {
                $data["Vehicle_Img.$index"] = $file;
            }

            // Validate the request
            $validator = Validator::make($data, [
                'SU_Id'        => 'nullable|exists:soc_units,SU_ID',
                'SocId'        => 'nullable|exists:society_master,SocId',
                'SCP_Id'       => 'nullable|exists:parking_masters,SCP_Id',
                'VehicleRegNo' => 'required|unique:vehicles,VehicleRegNo|max:25',
                'Status'       => 'required|in:Active,Inactive',
                'ActiveDate'   => 'nullable|date_format:Y-m-d H:i:s',
                'Vehicle_Type' => 'nullable|string|max:50',
                'Vehicle_Model' => 'nullable|string|max:50',
                'Vehicle_Brand' => 'nullable|string|max:50',
                'Vehicle_Img.*' => 'nullable|file|image|mimes:jpeg,png,jpg|max:10240'
            ]);

            if ($validator->fails()) {
                Log::error('Validation failed:', $validator->errors()->toArray());
                return response()->json($validator->errors(), 422);
            }

            $basePath = public_path('storage');
            $imageUrls = [];
            if ($request->hasFile('Vehicle_Img')) {
                foreach ($request->file('Vehicle_Img') as $index => $image) {
                    Log::info('Processing image:', ['index' => $index, 'name' => $image->getClientOriginalName()]);
                    $originalName = $image->getClientOriginalName();
                    $filePath = 'images/vehicles/' . $originalName;
                    $destinationPath = $basePath . '/' . $filePath;

                    if (file_exists($destinationPath)) {
                        $filenameWithoutExt = pathinfo($originalName, PATHINFO_FILENAME);
                        $extension = $image->getClientOriginalExtension();
                        $newFileName = $filenameWithoutExt . '_' . time() . '.' . $extension;
                        $filePath = 'images/vehicles/' . $newFileName;
                        $destinationPath = $basePath . '/' . $filePath;
                    }

                    if (!file_exists(dirname($destinationPath))) {
                        Log::info('Creating directory:', ['path' => dirname($destinationPath)]);
                        mkdir(dirname($destinationPath), 0755, true);
                    }

                    Log::info('Moving image:', ['from' => $image->getPathname(), 'to' => $destinationPath]);
                    $image->move(dirname($destinationPath), basename($filePath));

                    $imageUrls[] = url('storage/' . $filePath);
                }
            }

            // Convert ActiveDate from IST to UTC for storage
            $activeDate = $request->ActiveDate
                ? Carbon::createFromFormat('Y-m-d H:i:s', $request->ActiveDate, 'Asia/Kolkata')->setTimezone('UTC')
                : null;

            $vehicleData = [
                'SU_Id'        => $request->SU_Id,
                'SocId'        => $request->SocId,
                'SCP_Id'       => $request->SCP_Id,
                'VehicleRegNo' => $request->VehicleRegNo,
                'Status'       => $request->Status,
                'ActiveDate'   => $activeDate,
                'Vehicle_Type' => $request->Vehicle_Type,
                'Vehicle_Model' => $request->Vehicle_Model,
                'Vehicle_Brand' => $request->Vehicle_Brand,
                'Vehicle_Img'  => !empty($imageUrls) ? json_encode($imageUrls, JSON_UNESCAPED_SLASHES) : null,
                'CreatedAt'    => now(),
            ];
            Log::info('Creating vehicle:', $vehicleData);

            $vehicle = VehicleMaster::create($vehicleData);

            if ($vehicle->Vehicle_Img) {
                $imageUrls = is_string($vehicle->Vehicle_Img) ? json_decode($vehicle->Vehicle_Img, true) : $vehicle->Vehicle_Img;
                if (is_array($imageUrls)) {
                    $vehicle->Vehicle_Img = array_map(function ($url) {
                        return str_replace('\/', '/', $url);
                    }, $imageUrls);
                } else {
                    $vehicle->Vehicle_Img = [];
                }
            }
            // Convert ActiveDate to IST for response
            if ($vehicle->ActiveDate) {
                $vehicle->ActiveDate = Carbon::parse($vehicle->ActiveDate)->setTimezone('Asia/Kolkata')->format('Y-m-d H:i:s');
            }

            return response()->json($vehicle, 201);
        } catch (\Exception $e) {
            Log::error('Store error:', ['message' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json(['error' => 'Internal Server Error', 'message' => $e->getMessage()], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            Log::info('Update request received:', $request->all());
            Log::info('Files:', $request->file() ? array_keys($request->file()) : []);

            $vehicle = VehicleMaster::find($id);
            if (!$vehicle) {
                return response()->json(['message' => 'Vehicle not found'], 404);
            }

            // Prepare data for validation, excluding Vehicle_Img to avoid string validation
            $data = $request->except('Vehicle_Img');
            // Add only valid files for validation
            $files = $request->file('Vehicle_Img', []);
            foreach ($files as $index => $file) {
                $data["Vehicle_Img.$index"] = $file;
            }

            // Validate non-file fields and only actual file uploads
            $validator = Validator::make($data, [
                'SU_Id'        => 'sometimes|nullable|exists:soc_units,SU_ID',
                'SocId'        => 'sometimes|nullable|exists:society_master,SocId',
                'SCP_Id'       => 'sometimes|nullable|exists:parking_masters,SCP_Id',
                'VehicleRegNo' => 'sometimes|unique:vehicles,VehicleRegNo,' . $id . ',Vehicle_Id|max:25',
                'Status'       => 'sometimes|in:Active,Inactive',
                'ActiveDate'   => 'sometimes|nullable|date_format:Y-m-d H:i:s',
                'Vehicle_Type' => 'sometimes|nullable|string|max:50',
                'Vehicle_Model' => 'sometimes|nullable|string|max:50',
                'Vehicle_Brand' => 'sometimes|nullable|string|max:50',
                'Vehicle_Img.*' => 'nullable|file|image|mimes:jpeg,png,jpg|max:10240'
            ]);

            if ($validator->fails()) {
                Log::error('Validation failed:', $validator->errors()->toArray());
                return response()->json($validator->errors(), 422);
            }

            $validatedData = $request->except('Vehicle_Img');
            $existingImageUrls = $vehicle->Vehicle_Img ? (is_string($vehicle->Vehicle_Img) ? json_decode($vehicle->Vehicle_Img, true) : $vehicle->Vehicle_Img) : [];
            $basePath = public_path('storage');

            // Handle specific image updates
            $newImageUrls = $existingImageUrls;
            foreach ($files as $index => $image) {
                if ($image && $image->isValid()) {
                    Log::info('Processing image update:', ['index' => $index, 'name' => $image->getClientOriginalName()]);

                    // Delete old image if it exists at this index
                    if (isset($existingImageUrls[$index])) {
                        $oldUrl = $existingImageUrls[$index];
                        $oldPath = str_replace(url('storage/'), '', $oldUrl);
                        $fullOldPath = $basePath . '/' . $oldPath;
                        if (file_exists($fullOldPath)) {
                            unlink($fullOldPath);
                            Log::info('Deleted old image:', ['path' => $fullOldPath]);
                        }
                    }

                    // Store new image
                    $originalName = $image->getClientOriginalName();
                    $filePath = 'images/vehicles/' . $originalName;
                    $destinationPath = $basePath . '/' . $filePath;

                    if (file_exists($destinationPath)) {
                        $filenameWithoutExt = pathinfo($originalName, PATHINFO_FILENAME);
                        $extension = $image->getClientOriginalExtension();
                        $newFileName = $filenameWithoutExt . '_' . time() . '.' . $extension;
                        $filePath = 'images/vehicles/' . $newFileName;
                        $destinationPath = $basePath . '/' . $filePath;
                    }

                    if (!file_exists(dirname($destinationPath))) {
                        Log::info('Creating directory:', ['path' => dirname($destinationPath)]);
                        mkdir(dirname($destinationPath), 0755, true);
                    }

                    Log::info('Moving image:', ['from' => $image->getPathname(), 'to' => $destinationPath]);
                    $image->move(dirname($destinationPath), basename($filePath));

                    // Update the specific index
                    $newImageUrls[$index] = url('storage/' . $filePath);
                }
            }

            // Update Vehicle_Img only if new images were processed
            if ($newImageUrls !== $existingImageUrls) {
                $validatedData['Vehicle_Img'] = json_encode($newImageUrls, JSON_UNESCAPED_SLASHES);
            }

            // Convert ActiveDate from IST to UTC for storage
            if ($request->has('ActiveDate')) {
                $validatedData['ActiveDate'] = $request->ActiveDate
                    ? Carbon::createFromFormat('Y-m-d H:i:s', $request->ActiveDate, 'Asia/Kolkata')->setTimezone('UTC')
                    : null;
            }

            Log::info('Updating vehicle:', $validatedData);
            $vehicle->update($validatedData);

            // Prepare response
            if ($vehicle->Vehicle_Img) {
                $imageUrls = is_string($vehicle->Vehicle_Img) ? json_decode($vehicle->Vehicle_Img, true) : $vehicle->Vehicle_Img;
                if (is_array($imageUrls)) {
                    $vehicle->Vehicle_Img = array_map(function ($url) {
                        return str_replace('\/', '/', $url);
                    }, $imageUrls);
                } else {
                    $vehicle->Vehicle_Img = [];
                }
            }
            // Convert ActiveDate to IST for response
            if ($vehicle->ActiveDate) {
                $vehicle->ActiveDate = Carbon::parse($vehicle->ActiveDate)->setTimezone('Asia/Kolkata')->format('Y-m-d H:i:s');
            }

            return response()->json($vehicle);
        } catch (\Exception $e) {
            Log::error('Update error:', ['message' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json(['error' => 'Internal Server Error', 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * Delete specific images from a VehicleMaster record by Vehicle_Id.
     */

    public function deleteImages(Request $request, $id)
    {
        try {
            $vehicle = VehicleMaster::with(['unit.socBlock.sociDivision', 'society', 'parkingMaster'])->find($id);
            if (!$vehicle) {
                return response()->json(['message' => 'Vehicle not found'], 404);
            }

            // Log the incoming request data for debugging
            Log::debug('Delete images request data:', $request->all());

            // Validate the request
            $validator = Validator::make($request->all(), [
                'indices' => 'required_without:paths|array',
                'indices.*' => 'integer|min:0',
                'paths' => 'required_without:indices|array',
                'paths.*' => 'string|regex:/^images\/vehicles\/.+\.(jpg|jpeg|png)$/i',
            ], [
                'indices.required_without' => 'Either indices or paths must be provided.',
                'paths.required_without' => 'Either indices or paths must be provided.',
                'paths.*.regex' => 'Each path must be a valid image path starting with images/vehicles/ and ending with .jpg, .jpeg, or .png.'
            ]);

            if ($validator->fails()) {
                Log::error('Validation failed:', $validator->errors()->toArray());
                return response()->json($validator->errors(), 422);
            }

            // Get current image URLs
            $imageUrls = $vehicle->Vehicle_Img ? (is_string($vehicle->Vehicle_Img) ? json_decode($vehicle->Vehicle_Img, true) : $vehicle->Vehicle_Img) : [];
            if (empty($imageUrls)) {
                return response()->json(['message' => 'No images to delete'], 400);
            }

            // Determine images to delete
            $imagesToDelete = [];
            $newImageUrls = $imageUrls;

            if ($request->has('indices')) {
                $indices = array_unique($request->input('indices'));
                foreach ($indices as $index) {
                    if (isset($imageUrls[$index])) {
                        $imagesToDelete[] = $imageUrls[$index];
                        unset($newImageUrls[$index]);
                    } else {
                        Log::warning("Invalid index provided for deletion: {$index}");
                    }
                }
            } elseif ($request->has('paths')) {
                $paths = array_unique($request->input('paths'));
                foreach ($paths as $path) {
                    $fullUrl = url('storage/' . $path);
                    if (in_array($fullUrl, $imageUrls)) {
                        $imagesToDelete[] = $fullUrl;
                        $newImageUrls = array_diff($newImageUrls, [$fullUrl]);
                    } else {
                        Log::warning("Invalid path provided for deletion: {$path}");
                    }
                }
            }

            // Delete images from storage
            $basePath = public_path('storage');
            foreach ($imagesToDelete as $url) {
                $path = str_replace(url('storage/'), '', $url);
                $fullPath = $basePath . '/' . $path;
                if (file_exists($fullPath)) {
                    unlink($fullPath);
                    Log::info('Deleted image', ['path' => $path]);
                }
            }

            // Update the record with remaining images
            $vehicle->update([
                'Vehicle_Img' => !empty($newImageUrls) ? json_encode(array_values($newImageUrls), JSON_UNESCAPED_SLASHES) : null,
            ]);

            // Prepare response
            if ($vehicle->Vehicle_Img) {
                $imageUrls = is_string($vehicle->Vehicle_Img) ? json_decode($vehicle->Vehicle_Img, true) : $vehicle->Vehicle_Img;
                if (is_array($imageUrls)) {
                    $vehicle->Vehicle_Img = array_map(function ($url) {
                        return str_replace('\/', '/', $url);
                    }, $imageUrls);
                } else {
                    $vehicle->Vehicle_Img = [];
                }
            }
            // Convert ActiveDate to IST for response
            if ($vehicle->ActiveDate) {
                $vehicle->ActiveDate = Carbon::parse($vehicle->ActiveDate)->setTimezone('Asia/Kolkata')->format('Y-m-d H:i:s');
            }

            return response()->json($vehicle, 200);
        } catch (\Exception $e) {
            Log::error('Delete images error:', ['message' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json(['error' => 'Internal Server Error', 'message' => $e->getMessage()], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $vehicle = VehicleMaster::find($id);
            if (!$vehicle) {
                return response()->json(['message' => 'Vehicle not found'], 404);
            }

            if ($vehicle->Vehicle_Img) {
                $imageUrls = is_string($vehicle->Vehicle_Img) ? json_decode($vehicle->Vehicle_Img, true) : $vehicle->Vehicle_Img;
                if (is_array($imageUrls)) {
                    $basePath = public_path('storage');
                    foreach ($imageUrls as $url) {
                        $path = str_replace(url('storage/'), '', $url);
                        $fullPath = $basePath . '/' . $path;
                        if (file_exists($fullPath)) {
                            unlink($fullPath);
                        }
                    }
                }
            }

            $vehicle->delete();
            return response()->json(['message' => 'Vehicle deleted successfully']);
        } catch (\Exception $e) {
            Log::error('Destroy error:', ['message' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json(['error' => 'Internal Server Error', 'message' => $e->getMessage()], 500);
        }
    }
}
