<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\RoleMaster;
use App\Models\RoleMapping;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class UserController extends Controller
{
    public function index()
    {
        try {
            // Fetch all users with their role mappings, associated roles, and society, excluding SuperAdmin (RoleId 1)
            $users = User::with(['roleMappings.role', 'society'])
                ->whereDoesntHave('roleMappings', function ($query) {
                    $query->where('Role_Id', 1);
                })
                ->get();

            if ($users->isEmpty()) {
                return response()->json([
                    'status' => 'success',
                    'message' => 'No users found',
                    'data' => []
                ], 200);
            }

            return response()->json([
                'status' => 'success',
                'message' => 'Users retrieved successfully',
                'data' => $users
            ], 200);
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Error fetching users in index', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while retrieving users.'
            ], 500);
        }
    }

    public function show($id)
    {
        try {
            // Fetch a single user by U_Id with their role mappings, associated roles, and society
            $user = User::with(['roleMappings.role', 'society'])->find($id);

            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'User not found'
                ], 404);
            }

            // Check if the user has RoleId 1 (SuperAdmin)
            $isSuperAdmin = $user->roleMappings->contains(function ($roleMapping) {
                return $roleMapping->Role_Id == 1;
            });

            if ($isSuperAdmin) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'SuperAdmin user cannot be viewed.'
                ], 403);
            }

            return response()->json([
                'status' => 'success',
                'message' => 'User retrieved successfully',
                'data' => $user
            ], 200);
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Error fetching user in show', [
                'U_Id' => $id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while retrieving the user.'
            ], 500);
        }
    }

    public function store(Request $request)
    {
        // Extract Role_Id from the request, default to 4 if not provided
        $requestedRoleId = $request->input('Role_Id', 4);

        // Check if the requested Role_Id exists in RoleMaster
        $roleExists = RoleMaster::where('Role_Id', $requestedRoleId)->exists();
        if (!$roleExists) {
            return response()->json([
                'status' => 'error',
                'message' => "Cannot create user: Role with ID $requestedRoleId does not exist."
            ], 403);
        }

        // Prevent creating a new SuperAdmin (RoleId 1) if one already exists
        if ($requestedRoleId == 1) {
            $superAdminExists = RoleMapping::where('Role_Id', 1)->exists();
            if ($superAdminExists) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Cannot create user: A SuperAdmin (RoleId 1) already exists.'
                ], 403);
            }
        }

        $validator = Validator::make($request->all(), [
            'U_Name'     => 'required|string|max:50',
            'U_Email'    => 'nullable|email|unique:users_master,U_Email|max:35',
            'U_Password' => 'required|string|min:8',
            'U_Status'   => 'nullable|string|max:1|in:A,I',
            'U_Gender'   => 'nullable|string|max:1|in:M,F,O',
            'U_Mobile'   => 'required|digits:10|unique:users_master,U_Mobile',
            'U_DOB'      => 'nullable|date_format:d-m-Y|before:today',
            'U_Image'    => 'nullable|string',
            'SocId'      => 'nullable|exists:society_master,SocId', // Added SocId validation
        ], [
            'U_Name.required' => 'User name is required.',
            'U_Name.max' => 'User name must not exceed 50 characters.',
            'U_Email.email' => 'Email must be a valid email address.',
            'U_Email.unique' => 'This email is already registered.',
            'U_Email.max' => 'Email must not exceed 35 characters.',
            'U_Password.required' => 'Password is required.',
            'U_Password.min' => 'Password must be at least 8 characters long.',
            'U_Status.in' => 'Status must be either A (Active) or I (Inactive).',
            'U_Gender.in' => 'Gender must be M (Male), F (Female), or O (Other).',
            'U_Mobile.required' => 'Mobile number is required.',
            'U_Mobile.digits' => 'Mobile number must be exactly 10 digits.',
            'U_Mobile.unique' => 'This mobile number is already registered.',
            'U_DOB.date_format' => 'Date of birth must be in dd-mm-yyyy format (e.g., 15-08-1990).',
            'U_DOB.before' => 'Date of birth must be a date before today (11-06-2025).',
            'U_Image.string' => 'Image must be a valid base64-encoded string.',
            'SocId.exists' => 'The selected society ID does not exist.', // Added SocId error message
        ]);

        if ($validator->fails()) {
            // Log invalid U_DOB for debugging
            if ($request->has('U_DOB') && $validator->errors()->has('U_DOB')) {
                \Illuminate\Support\Facades\Log::warning('Invalid U_DOB format in store', [
                    'U_DOB' => $request->input('U_DOB'),
                    'errors' => $validator->errors()->get('U_DOB')
                ]);
            }

            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $validatedData = $validator->validated();

        // Sanitize string inputs
        $validatedData['U_Name'] = trim($validatedData['U_Name']);
        if (!empty($validatedData['U_Email'])) {
            $validatedData['U_Email'] = trim($validatedData['U_Email']);
        }

        // Set default U_Status = "A" if not provided
        $validatedData['U_Status'] = $validatedData['U_Status'] ?? 'A';

        // Handle image upload (base64)
        if (!empty($validatedData['U_Image'])) {
            $imageData = base64_decode($validatedData['U_Image']);
            $imageName = time() . '.png';
            $imagePath = public_path('storage/user_images/' . $imageName);
            file_put_contents($imagePath, $imageData);
            $validatedData['U_Image'] = url('storage/user_images/' . $imageName);
        }

        // Hash the password
        $validatedData['U_Password'] = Hash::make($validatedData['U_Password']);

        // Convert U_DOB to MySQL format (Y-m-d)
        if (!empty($validatedData['U_DOB'])) {
            try {
                $validatedData['U_DOB'] = Carbon::createFromFormat('d-m-Y', $validatedData['U_DOB'], 'Asia/Kolkata')
                    ->startOfDay()
                    ->format('Y-m-d');
            } catch (\Exception $e) {
                \Illuminate\Support\Facades\Log::warning('Failed to parse U_DOB in store', [
                    'U_DOB' => $validatedData['U_DOB'],
                    'error' => $e->getMessage()
                ]);
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid date format for date of birth. Use dd-mm-yyyy (e.g., 15-08-1990).'
                ], 422);
            }
        }

        try {
            // Create the user
            $user = User::create($validatedData);

            // Create RoleMapping entry with the requested Role_Id and Default_Rid = 1
            $roleMapping = RoleMapping::create([
                'U_Id' => $user->U_Id,
                'Role_Id' => $requestedRoleId,
                'Default_Rid' => 1
            ]);

            // Load role mapping data
            $user->load(['roleMappings' => function ($query) {
                $query->with('role');
            }]);

            return response()->json([
                'status' => 'success',
                'message' => "User created successfully with role (Role_Id = $requestedRoleId)",
                'data' => [
                    'user' => $user,
                    'role_mapping' => $roleMapping
                ]
            ], 201);
        } catch (\Exception $e) {
            // Log the error for debugging
            \Illuminate\Support\Facades\Log::error('Error creating user or role mapping', [
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while creating the user or role mapping.'
            ], 500);
        }
    }

    public function update(Request $request, $id)
    {
        $user = User::find($id);
        if (!$user) {
            return response()->json(['message' => 'User not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'U_Name'     => 'sometimes|string|max:50',
            'U_Email'    => 'nullable|email|unique:users_master,U_Email,' . $id . ',U_Id|max:35',
            'U_Password' => 'sometimes|string|min:8',
            'U_Status'   => 'nullable|string|max:1|in:A,I',
            'U_Gender'   => 'nullable|string|max:1|in:M,F,O',
            'U_Mobile'   => 'sometimes|digits:10|unique:users_master,U_Mobile,' . $id . ',U_Id',
            'U_DOB'      => 'nullable|date_format:d-m-Y|before:today',
            'U_Image'    => 'nullable|string',
            'SocId'      => 'nullable|exists:society_master,SocId', // Added SocId validation
        ], [
            'U_Name.max' => 'User name must not exceed 50 characters.',
            'U_Email.email' => 'Email must be a valid email address.',
            'U_Email.unique' => 'This email is already registered.',
            'U_Email.max' => 'Email must not exceed 35 characters.',
            'U_Password.min' => 'Password must be at least 8 characters long.',
            'U_Status.in' => 'Status must be either A (Active) or I (Inactive).',
            'U_Gender.in' => 'Gender must be M (Male), F (Female), or O (Other).',
            'U_Mobile.digits' => 'Mobile number must be exactly 10 digits.',
            'U_Mobile.unique' => 'This mobile number is already registered.',
            'U_DOB.date_format' => 'Date of birth must be in dd-mm-yyyy format (e.g., 15-08-1990).',
            'U_DOB.before' => 'Date of birth must be a date before today (11-06-2025).',
            'U_Image.string' => 'Image must be a valid base64-encoded string.',
            'SocId.exists' => 'The selected society ID does not exist.', // Added SocId error message
        ]);

        if ($validator->fails()) {
            // Log invalid U_DOB for debugging
            if ($request->has('U_DOB') && $validator->errors()->has('U_DOB')) {
                \Illuminate\Support\Facades\Log::warning('Invalid U_DOB format in update', [
                    'U_DOB' => $request->input('U_DOB'),
                    'errors' => $validator->errors()->get('U_DOB')
                ]);
            }

            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $validatedData = $request->only([
            'U_Name',
            'U_Email',
            'U_Status',
            'U_Gender',
            'U_Mobile',
            'U_DOB',
            'U_Image',
            'SocId', // Added SocId
        ]);

        // Handle password if provided
        if ($request->has('U_Password') && $request->filled('U_Password')) {
            $validatedData['U_Password'] = Hash::make($request->input('U_Password'));
        }

        // Handle date of birth formatting
        if (isset($validatedData['U_DOB']) && $validatedData['U_DOB']) {
            try {
                $validatedData['U_DOB'] = Carbon::createFromFormat('d-m-Y', $validatedData['U_DOB'], 'Asia/Kolkata')
                    ->startOfDay()
                    ->format('Y-m-d');
            } catch (\Exception $e) {
                \Illuminate\Support\Facades\Log::warning('Failed to parse U_DOB in update', [
                    'U_DOB' => $validatedData['U_DOB'],
                    'error' => $e->getMessage()
                ]);
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid date format for date of birth. Use dd-mm-yyyy (e.g., 15-08-1990).'
                ], 422);
            }
        }

        $user->update($validatedData);
        return response()->json([
            'status' => 'success',
            'message' => 'User updated successfully',
            'data' => $user
        ]);
    }

    public function destroy($id)
    {
        $user = User::find($id);
        if (!$user) {
            return response()->json(['message' => 'User not found'], 404);
        }

        $user->delete();
        return response()->json(['message' => 'User deleted successfully']);
    }

    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'U_Email'    => 'sometimes|email|max:35',
            'U_Mobile'   => 'sometimes|digits:10',
            'U_Password' => 'required|string|min:8',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        $user = null;
        if ($request->U_Email) {
            $user = User::where('U_Email', $request->U_Email)->first();
        } elseif ($request->U_Mobile) {
            $user = User::where('U_Mobile', $request->U_Mobile)->first();
        }

        if (!$user) {
            return response()->json(['status' => 'error', 'message' => 'User not found'], 404);
        }

        if ($user->U_Status !== 'A') {
            return response()->json([
                'status' => 'error',
                'message' => 'Account is inactive. Please contact administrator.'
            ], 403);
        }

        if (!Hash::check(trim($request->U_Password), $user->U_Password)) {
            return response()->json(['status' => 'error', 'message' => 'Invalid credentials'], 401);
        }

        $token = $user->createToken('User Authentication')->plainTextToken;
        return response()->json([
            'status' => 'success',
            'message' => 'Login successful',
            'user' => $user,
            'token' => $token,
        ]);
    }
}
