<?php

namespace App\Http\Controllers;

use App\Models\UnitType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class UnitTypeController extends Controller
{
    public function index()
    {
        $unitTypes = UnitType::with('society')->get();
        return response()->json([
            'status' => 'success',
            'message' => 'Unit types retrieved successfully',
            'data' => $unitTypes
        ], 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'UT_Name' => 'nullable|string|max:10',
            'UT_Size' => 'nullable|integer|min:0',
            'SizeUnit' => 'nullable|string|max:10',
            'SocId' => 'nullable|exists:society_master,SocId',
        ], [
            'UT_Name.max' => 'Unit name must not exceed 10 characters.',
            'UT_Size.integer' => 'Unit size must be a valid number.',
            'UT_Size.min' => 'Unit size cannot be negative.',
            'SizeUnit.in' => 'Size unit must be S (square feet) or M (square meters).',
            'SocId.exists' => 'The selected society ID is invalid.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $unitType = UnitType::create($validator->validated());
            return response()->json([
                'status' => 'success',
                'message' => 'Unit type created successfully',
                'data' => $unitType
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to create unit type: ' . $e->getMessage(),
                'error_code' => $e->getCode()
            ], 500);
        }
    }

    public function show($id)
    {
        $unitType = UnitType::with('society')->find($id);
        if (!$unitType) {
            return response()->json([
                'status' => 'error',
                'message' => 'Unit type not found'
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Unit type retrieved successfully',
            'data' => $unitType
        ], 200);
    }

    public function update(Request $request, $id)
    {
        $unitType = UnitType::find($id);
        if (!$unitType) {
            return response()->json([
                'status' => 'error',
                'message' => 'Unit type not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'UT_Name' => 'nullable|string|max:10',
            'UT_Size' => 'nullable|integer|min:0',
            'SizeUnit' => 'nullable|string|max:10',
            'SocId' => 'nullable|exists:society_master,SocId',
        ], [
            'UT_Name.max' => 'Unit name must not exceed 10 characters.',
            'UT_Size.integer' => 'Unit size must be a valid number.',
            'UT_Size.min' => 'Unit size cannot be negative.',
            'SizeUnit.in' => 'Size unit must be S (square feet) or M (square meters).',
            'SocId.exists' => 'The selected society ID is invalid.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $unitType->update($request->only(['UT_Name', 'UT_Size', 'SizeUnit', 'SocId']));
            return response()->json([
                'status' => 'success',
                'message' => 'Unit type updated successfully',
                'data' => $unitType
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update unit type: ' . $e->getMessage(),
                'error_code' => $e->getCode()
            ], 500);
        }
    }

    public function destroy($id)
    {
        $unitType = UnitType::find($id);
        if (!$unitType) {
            return response()->json([
                'status' => 'error',
                'message' => 'Unit type not found'
            ], 404);
        }

        try {
            $unitType->delete();
            return response()->json([
                'status' => 'success',
                'message' => 'Unit type deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to delete unit type: ' . $e->getMessage(),
                'error_code' => $e->getCode()
            ], 500);
        }
    }
}
