<?php

namespace App\Http\Controllers;

use App\Models\SociType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class SociTypeController extends Controller
{
    public function index()
    {
        $sociTypes = SociType::all();
        return response()->json([
            'status' => 'success',
            'message' => 'Society types retrieved successfully',
            'data' => $sociTypes
        ], 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'SType_Name' => 'required|string|max:25|unique:soci_type,SType_Name',
            'Status' => 'required|string|max:1|in:A,I',
        ], [
            'SType_Name.required' => 'Society type name is required.',
            'SType_Name.max' => 'Society type name must not exceed 25 characters.',
            'SType_Name.unique' => 'This society type name already exists.',
            'Status.required' => 'Status is required.',
            'Status.in' => 'Status must be A (Active) or I (Inactive).',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $sociType = SociType::create($validator->validated());
            return response()->json([
                'status' => 'success',
                'message' => 'Society type created successfully',
                'data' => $sociType
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while creating the society type.'
            ], 500);
        }
    }

    public function show($id)
    {
        $sociType = SociType::find($id);
        if (!$sociType) {
            return response()->json([
                'status' => 'error',
                'message' => 'Society type not found'
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Society type retrieved successfully',
            'data' => $sociType
        ], 200);
    }

    public function update(Request $request, $id)
    {
        $sociType = SociType::find($id);
        if (!$sociType) {
            return response()->json([
                'status' => 'error',
                'message' => 'Society type not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'SType_Name' => 'sometimes|string|max:25|unique:soci_type,SType_Name,' . $id . ',St_Id',
            'Status' => 'sometimes|string|max:1|in:A,I',
        ], [
            'SType_Name.max' => 'Society type name must not exceed 25 characters.',
            'SType_Name.unique' => 'This society type name already exists.',
            'Status.in' => 'Status must be A (Active) or I (Inactive).',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $sociType->update($request->only(['SType_Name', 'Status']));
            return response()->json([
                'status' => 'success',
                'message' => 'Society type updated successfully',
                'data' => $sociType
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while updating the society type.'
            ], 500);
        }
    }

    public function destroy($id)
    {
        $sociType = SociType::find($id);
        if (!$sociType) {
            return response()->json([
                'status' => 'error',
                'message' => 'Society type not found'
            ], 404);
        }

        try {
            $sociType->delete();
            return response()->json([
                'status' => 'success',
                'message' => 'Society type deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while deleting the society type.'
            ], 500);
        }
    }
}
