<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\SociResoBook;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;

class SociResoBookController extends Controller
{
    /**
     * Get all records.
     */
    public function index(Request $request)
    {
        // Get SocId from query parameter
        $socId = $request->query('SocId');

        // If SocId is provided, filter the records by SocId and load the related SocietyMaster
        if ($socId) {
            $sociResoBooks = SociResoBook::where('SocId', $socId)->with('society')->get();
        } else {
            // If no SocId is provided, get all records with related SocietyMaster data
            $sociResoBooks = SociResoBook::with('society')->get();
        }

        // Loop through all records and add the full URL to Sr_File
        $sociResoBooks->each(function($sociResoBook) {
            $sociResoBook->Sr_File = url('storage/uploads/' . $sociResoBook->Sr_File);
        });

        return response()->json($sociResoBooks);
    }

    /**
     * Create a new record with file upload.
     */
    public function store(Request $request)
    {
        // Validation rules - allow all file types with a maximum size of 10MB
        $validator = Validator::make($request->all(), [
            'SocId'         => 'required|integer|exists:society_master,SocId',
            'Sr_Subject'    => 'required|string|max:255',
            'Sr_Date'       => 'required|date',
            'Sr_Place'      => 'required|string|max:255',
            'Sr_Time'       => 'required|date_format:Y-m-d H:i:s',
            'Sr_Description' => 'required|string|max:500',
            'Sr_File'       => 'nullable|file|max:10240', // Accept all file types, up to 10MB
        ]);

        // Handle file upload
        $filePath = null;
        $fileName = null; // Variable to store the original file name
        if ($request->hasFile('Sr_File')) {
            $file = $request->file('Sr_File');
            $fileName = $file->getClientOriginalName(); // Get the original file name
            $filePath = $file->storeAs('uploads', $fileName, 'public'); // Store file with original name
        }

        // Create the record
        $sociResoBook = SociResoBook::create([
            'SocId'         => $request->input('SocId'),
            'Sr_Subject'    => $request->input('Sr_Subject'),
            'Sr_Date'       => $request->input('Sr_Date'),
            'Sr_Place'      => $request->input('Sr_Place'),
            'Sr_Time'       => $request->input('Sr_Time'),
            'Sr_Description' => $request->input('Sr_Description'),
            'Sr_File'       => $fileName, // Save the original file name
        ]);

        // Update the Sr_File field with the full URL
        $sociResoBook->Sr_File = url('storage/uploads/' . $fileName);

        return response()->json($sociResoBook, 201);
    }

    /**
     * Get a specific record.
     */
    public function show($id)
    {
        $sociResoBook = SociResoBook::find($id);

        if (!$sociResoBook) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        return response()->json($sociResoBook);
    }

    /**
     * Update a record with file upload.
     */
    public function update(Request $request, $id)
    {
        $sociResoBook = SociResoBook::find($id);

        if (!$sociResoBook) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        // Validation rules - allow all file types with a maximum size of 10MB
        $validator = Validator::make($request->all(), [
            'SocId'         => 'sometimes|required|integer|exists:society_master,SocId',
            'Sr_Subject'    => 'sometimes|required|string|max:255',
            'Sr_Date'       => 'sometimes|required|date',
            'Sr_Place'      => 'sometimes|required|string|max:255',
            'Sr_Time'       => 'sometimes|required|date_format:Y-m-d H:i:s',
            'Sr_Description' => 'sometimes|required|string|max:500',
            'Sr_File'       => 'nullable|file|max:10240', // Accept all file types, up to 10MB
        ]);

        // If validation fails, return a response with errors
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // Handle file upload
        if ($request->hasFile('Sr_File')) {
            // Delete old file if exists
            if ($sociResoBook->Sr_File) {
                Storage::disk('public')->delete('uploads/' . $sociResoBook->Sr_File); // Delete the old file
            }

            // Store new file with original name
            $file = $request->file('Sr_File');
            $fileName = $file->getClientOriginalName(); // Get the original file name
            $filePath = $file->storeAs('uploads', $fileName, 'public'); // Store file with original name
            $sociResoBook->Sr_File = $fileName; // Save the new file name in the database
        }

        // Update record
        $sociResoBook->update($request->except(['Sr_File']) + ['Sr_File' => $sociResoBook->Sr_File]);

        // Update the Sr_File field with the full URL
        $sociResoBook->Sr_File = url('storage/uploads/' . $sociResoBook->Sr_File);

        return response()->json($sociResoBook);
    }

    /**
     * Delete a record.
     */
    public function destroy($id)
    {
        $sociResoBook = SociResoBook::find($id);

        if (!$sociResoBook) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        // Delete file if exists
        if ($sociResoBook->Sr_File) {
            Storage::disk('public')->delete('uploads/' . $sociResoBook->Sr_File);
        }

        $sociResoBook->delete();

        return response()->json(['message' => 'Deleted successfully']);
    }
}
