<?php

namespace App\Http\Controllers;

use App\Models\SociResAttn;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class SociResAttnController extends Controller
{
    /**
     * Get all SociResAttn records.
     */
    public function index()
    {
        $attns = SociResAttn::with(['sociRes', 'user'])->get();
        return response()->json($attns);
    }

    /**
     * Get a specific SociResAttn record by SRA_ID.
     */
    public function show($id)
    {
        $attn = SociResAttn::with(['sociRes', 'user'])->find($id);

        if (!$attn) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        return response()->json($attn);
    }

    /**
     * Create a new SociResAttn record.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'SR_Id' => 'nullable|integer|exists:soci_reso_books,SR_Id',
            'U_Id' => 'required|integer|exists:users_master,U_Id',
            'SRU_Conf' => 'nullable|string|in:Y,N|max:1',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        DB::beginTransaction(); // Start transaction

        try {
            $validated = $validator->validated();

            // Create SociResAttn record
            $attn = SociResAttn::create($validated);

            DB::commit(); // Commit transaction

            // Eager load the related soci res and user
            $attn->load('sociRes', 'user');

            return response()->json([
                'message' => 'Society resident attention registered successfully!',
                'attn' => $attn
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack(); // Rollback on failure
            return response()->json(['error' => 'Society resident attention registration failed!', 'details' => $e->getMessage()], 500);
        }
    }

    /**
     * Update a SociResAttn record by SRA_ID.
     */
    public function update(Request $request, $id)
    {
        $attn = SociResAttn::find($id);

        if (!$attn) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'SR_Id' => 'nullable|integer|exists:soci_reso_books,SR_Id',
            'U_Id' => 'nullable|integer|exists:users_master,U_Id',
            'SRU_Conf' => 'nullable|string|in:Y,N|max:1',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $validated = array_filter($validator->validated());

        $attn->update($validated);

        $attn->load('sociRes', 'user');

        return response()->json($attn);
    }

    /**
     * Delete a SociResAttn record by SRA_ID.
     */
    public function destroy($id)
    {
        $attn = SociResAttn::find($id);

        if (!$attn) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        $attn->delete();
        return response()->json(['message' => 'Deleted successfully']);
    }
}
