<?php

namespace App\Http\Controllers;

use App\Models\SociDivision;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class SociDivisionController extends Controller
{
    public function index()
    {
        $divisions = SociDivision::with('society')->get();
        return response()->json([
            'status' => 'success',
            'message' => 'Society divisions retrieved successfully',
            'data' => $divisions
        ], 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'Sd_Name' => 'required|string|max:25',
            'SocId' => 'required|exists:society_master,SocId',
        ], [
            'Sd_Name.required' => 'Division name is required.',
            'Sd_Name.max' => 'Division name must not exceed 25 characters.',
            'SocId.required' => 'Society ID is required.',
            'SocId.exists' => 'The selected society does not exist.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $division = SociDivision::create($validator->validated());
            $division->load('society');
            return response()->json([
                'status' => 'success',
                'message' => 'Society division created successfully',
                'data' => $division
            ], 201);
        } catch (\Exception $e) {
            Log::error('Error creating society division: ' . $e->getMessage(), [
                'request' => $request->all(),
                'exception' => $e->getTraceAsString()
            ]);
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to create society division: ' . $e->getMessage(),
                'error_code' => $e->getCode()
            ], 500);
        }
    }

    public function show($id)
    {
        $division = SociDivision::with('society')->find($id);
        if (!$division) {
            return response()->json([
                'status' => 'error',
                'message' => 'Society division not found'
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Society division retrieved successfully',
            'data' => $division
        ], 200);
    }

    public function update(Request $request, $id)
    {
        $division = SociDivision::find($id);
        if (!$division) {
            return response()->json([
                'status' => 'error',
                'message' => 'Society division not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'Sd_Name' => 'sometimes|string|max:25',
            'SocId' => 'sometimes|exists:society_master,SocId',
        ], [
            'Sd_Name.max' => 'Division name must not exceed 25 characters.',
            'SocId.exists' => 'The selected society does not exist.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $division->update($request->only(['Sd_Name', 'SocId']));
            $division->load('society');
            return response()->json([
                'status' => 'success',
                'message' => 'Society division updated successfully',
                'data' => $division
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error updating society division: ' . $e->getMessage(), [
                'request' => $request->all(),
                'exception' => $e->getTraceAsString()
            ]);
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update society division: ' . $e->getMessage(),
                'error_code' => $e->getCode()
            ], 500);
        }
    }

    public function destroy($id)
    {
        $division = SociDivision::find($id);
        if (!$division) {
            return response()->json([
                'status' => 'error',
                'message' => 'Society division not found'
            ], 404);
        }

        try {
            $division->delete();
            return response()->json([
                'status' => 'success',
                'message' => 'Society division deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error deleting society division: ' . $e->getMessage(), [
                'id' => $id,
                'exception' => $e->getTraceAsString()
            ]);
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to delete society division: ' . $e->getMessage(),
                'error_code' => $e->getCode()
            ], 500);
        }
    }
}
