<?php

namespace App\Http\Controllers;

use App\Models\SocBlock;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class SocBlockController extends Controller
{
    public function index()
    {
        $blocks = SocBlock::with(['division' => function ($query) {
            $query->with('society');
        }])->get();
        return response()->json([
            'status' => 'success',
            'message' => 'Society blocks retrieved successfully',
            'data' => $blocks
        ], 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'SB_Name' => 'required|string|max:25',
            'Sd_Id' => 'required|exists:soci_divisions,Sd_Id',
        ], [
            'SB_Name.required' => 'Block name is required.',
            'SB_Name.max' => 'Block name must not exceed 25 characters.',
            'Sd_Id.required' => 'Division ID is required.',
            'Sd_Id.exists' => 'The selected division does not exist.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $block = SocBlock::create($validator->validated());
            $block->load(['division' => function ($query) {
                $query->with('society');
            }]);
            return response()->json([
                'status' => 'success',
                'message' => 'Society block created successfully',
                'data' => $block
            ], 201);
        } catch (\Exception $e) {
            Log::error('Error creating society block: ' . $e->getMessage(), [
                'request' => $request->all(),
                'exception' => $e->getTraceAsString()
            ]);
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to create society block: ' . $e->getMessage(),
                'error_code' => $e->getCode()
            ], 500);
        }
    }

    public function show($id)
    {
        $block = SocBlock::with(['division' => function ($query) {
            $query->with('society');
        }])->find($id);
        if (!$block) {
            return response()->json([
                'status' => 'error',
                'message' => 'Society block not found'
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Society block retrieved successfully',
            'data' => $block
        ], 200);
    }

    public function update(Request $request, $id)
    {
        $block = SocBlock::find($id);
        if (!$block) {
            return response()->json([
                'status' => 'error',
                'message' => 'Society block not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'SB_Name' => 'sometimes|string|max:25',
            'Sd_Id' => 'sometimes|exists:soci_divisions,Sd_Id',
        ], [
            'SB_Name.max' => 'Block name must not exceed 25 characters.',
            'Sd_Id.exists' => 'The selected division does not exist.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $block->update($request->only(['SB_Name', 'Sd_Id']));
            $block->load(['division' => function ($query) {
                $query->with('society');
            }]);
            return response()->json([
                'status' => 'success',
                'message' => 'Society block updated successfully',
                'data' => $block
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error updating society block: ' . $e->getMessage(), [
                'request' => $request->all(),
                'exception' => $e->getTraceAsString()
            ]);
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update society block: ' . $e->getMessage(),
                'error_code' => $e->getCode()
            ], 500);
        }
    }

    public function destroy($id)
    {
        $block = SocBlock::find($id);
        if (!$block) {
            return response()->json([
                'status' => 'error',
                'message' => 'Society block not found'
            ], 404);
        }

        try {
            $block->delete();
            return response()->json([
                'status' => 'success',
                'message' => 'Society block deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error deleting society block: ' . $e->getMessage(), [
                'id' => $id,
                'exception' => $e->getTraceAsString()
            ]);
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to delete society block: ' . $e->getMessage(),
                'error_code' => $e->getCode()
            ], 500);
        }
    }
}
