<?php

namespace App\Http\Controllers;

use App\Models\PollBook;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PollBookController extends Controller
{
    /**
     * Get all PollBook records with related data.
     */
    public function index()
    {
        $pollBooks = PollBook::with(['pollCategory', 'society', 'user'])->get();
        return response()->json($pollBooks);
    }

    /**
     * Get a specific PollBook record by Poll_Id with related data.
     */
    public function show($id)
    {
        $pollBook = PollBook::with(['pollCategory', 'society', 'user'])->find($id);

        if (!$pollBook) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        return response()->json($pollBook);
    }

    /**
     * Create a new PollBook record.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'Pc_ID'      => 'required|integer|exists:poll_category,Pc_ID',
            'SocId'      => 'required|integer|exists:society_master,SocId',
            'U_Id'       => 'required|integer|exists:users_master,U_Id',
            'Title'      => 'required|string|max:25',
            'Description'=> 'required|string|max:250',
            'PollType'   => 'required|string|max:1',
            'Severity'   => 'required|string|max:1',
            'Status'     => 'required|string|max:1',
            'STDate'     => 'required|date_format:Y-m-d H:i:s',
            'EdDate'     => 'required|date_format:Y-m-d H:i:s',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $pollBook = PollBook::create([
            'Pc_ID'      => $request->input('Pc_ID'),
            'SocId'      => $request->input('SocId'),
            'U_Id'       => $request->input('U_Id'),
            'Title'      => $request->input('Title'),
            'Description'=> $request->input('Description'),
            'PollType'   => $request->input('PollType'),
            'Severity'   => $request->input('Severity'),
            'Status'     => $request->input('Status'),
            'STDate'     => $request->input('STDate'),
            'EdDate'     => $request->input('EdDate'),
        ]);

        return response()->json($pollBook, 201);
    }

    /**
     * Update a PollBook record by Poll_Id.
     */
    public function update(Request $request, $id)
    {
        $pollBook = PollBook::find($id);

        if (!$pollBook) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'Pc_ID'      => 'required|integer|exists:poll_category,Pc_ID',
            'SocId'      => 'required|integer|exists:society_master,SocId',
            'U_Id'       => 'required|integer|exists:users_master,U_Id',
            'Title'      => 'required|string|max:25',
            'Description'=> 'required|string|max:250',
            'PollType'   => 'required|string|max:1',
            'Severity'   => 'required|string|max:1',
            'Status'     => 'required|string|max:1',
            'STDate'     => 'required|date_format:Y-m-d H:i:s',
            'EdDate'     => 'required|date_format:Y-m-d H:i:s',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $pollBook->update([
            'Pc_ID'      => $request->input('Pc_ID'),
            'SocId'      => $request->input('SocId'),
            'U_Id'       => $request->input('U_Id'),
            'Title'      => $request->input('Title'),
            'Description'=> $request->input('Description'),
            'PollType'   => $request->input('PollType'),
            'Severity'   => $request->input('Severity'),
            'Status'     => $request->input('Status'),
            'STDate'     => $request->input('STDate'),
            'EdDate'     => $request->input('EdDate'),
        ]);

        return response()->json($pollBook);
    }

    /**
     * Delete a PollBook record by Poll_Id.
     */
    public function destroy($id)
    {
        $pollBook = PollBook::find($id);

        if (!$pollBook) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        $pollBook->delete();
        return response()->json(['message' => 'Deleted successfully']);
    }
}
