<?php

namespace App\Http\Controllers;

use App\Models\ParkingLevel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ParkingLevelController extends Controller
{
    /**
     * Display a listing of the parking levels.
     */
    public function index()
    {
        $parkingLevels = ParkingLevel::with('society')->get();
        return response()->json([
            'message' => 'Parking levels retrieved successfully',
            'data' => $parkingLevels,
        ], 200);
    }

    /**
     * Store a newly created parking level in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'Level_Desc' => 'required|string|max:100|unique:parking_levels,Level_Desc',
            'SocId' => 'nullable|exists:society_master,SocId',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'error' => $validator->errors(),
            ], 422);
        }

        $parkingLevel = ParkingLevel::create([
            'Level_Desc' => $request->Level_Desc,
            'SocId' => $request->SocId,
        ]);

        return response()->json([
            'message' => 'Parking level created successfully',
            'data' => $parkingLevel->load('society'),
        ], 201);
    }

    /**
     * Display the specified parking level.
     */
    public function show($id)
    {
        $parkingLevel = ParkingLevel::with('society')->find($id);

        if (!$parkingLevel) {
            return response()->json([
                'error' => 'Parking level not found',
            ], 404);
        }

        return response()->json([
            'message' => 'Parking level retrieved successfully',
            'data' => $parkingLevel,
        ], 200);
    }

    /**
     * Update the specified parking level in storage.
     */
    public function update(Request $request, $id)
    {
        $parkingLevel = ParkingLevel::find($id);

        if (!$parkingLevel) {
            return response()->json([
                'error' => 'Parking level not found',
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'Level_Desc' => 'required|string|max:100|unique:parking_levels,Level_Desc,' . $id . ',PL_Id',
            'SocId' => 'nullable|exists:society_master,SocId',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'error' => $validator->errors(),
            ], 422);
        }

        $parkingLevel->update([
            'Level_Desc' => $request->Level_Desc,
            'SocId' => $request->SocId,
        ]);

        return response()->json([
            'message' => 'Parking level updated successfully',
            'data' => $parkingLevel->load('society'),
        ], 200);
    }

    /**
     * Remove the specified parking level from storage.
     */
    public function destroy($id)
    {
        $parkingLevel = ParkingLevel::find($id);

        if (!$parkingLevel) {
            return response()->json([
                'error' => 'Parking level not found',
            ], 404);
        }

        $parkingLevel->delete();

        return response()->json([
            'message' => 'Parking level deleted successfully',
        ], 200);
    }
}
