<?php

namespace App\Http\Controllers;

use App\Models\PBType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PBTypeController extends Controller
{
    public function index()
    {
        $pbTypes = PBType::with('society')->get();
        return response()->json([
            'status' => 'success',
            'message' => 'Postbox types retrieved successfully',
            'data' => $pbTypes
        ], 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'Post_Box_Desc' => 'required|string|max:25',
            'SocId' => 'required|exists:society_master,SocId',
        ], [
            'Post_Box_Desc.required' => 'The postbox description is required.',
            'Post_Box_Desc.string' => 'The postbox description must be a string.',
            'Post_Box_Desc.max' => 'The postbox description may not exceed 25 characters.',
            'SocId.required' => 'The society ID is required.',
            'SocId.exists' => 'The selected society does not exist.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $data = $validator->validated();
            $pbType = PBType::create($data);
            $pbType->load('society');
            return response()->json([
                'status' => 'success',
                'message' => 'Postbox type created successfully',
                'data' => $pbType
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while creating the postbox type: ' . $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        $pbType = PBType::with('society')->find($id);
        if (!$pbType) {
            return response()->json([
                'status' => 'error',
                'message' => 'Postbox type not found'
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Postbox type retrieved successfully',
            'data' => $pbType
        ], 200);
    }

    public function update(Request $request, $id)
    {
        $pbType = PBType::find($id);
        if (!$pbType) {
            return response()->json([
                'status' => 'error',
                'message' => 'Postbox type not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'Post_Box_Desc' => 'required|string|max:25',
            'SocId' => 'required|exists:society_master,SocId',
        ], [
            'Post_Box_Desc.required' => 'The postbox description is required.',
            'Post_Box_Desc.string' => 'The postbox description must be a string.',
            'Post_Box_Desc.max' => 'The postbox description may not exceed 25 characters.',
            'SocId.required' => 'The society ID is required.',
            'SocId.exists' => 'The selected society does not exist.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $data = $validator->validated();
            $pbType->update($data);
            $pbType->load('society');
            return response()->json([
                'status' => 'success',
                'message' => 'Postbox type updated successfully',
                'data' => $pbType
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while updating the postbox type.'
            ], 500);
        }
    }

    public function destroy($id)
    {
        $pbType = PBType::find($id);
        if (!$pbType) {
            return response()->json([
                'status' => 'error',
                'message' => 'Postbox type not found'
            ], 404);
        }

        try {
            $pbType->delete();
            return response()->json([
                'status' => 'success',
                'message' => 'Postbox type deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while deleting the postbox type.'
            ], 500);
        }
    }
}
