<?php

namespace App\Http\Controllers;

use App\Models\PBAllotment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class PBAllotmentController extends Controller
{
    public function index()
    {
        $pbAllotments = PBAllotment::with(['pbUnit', 'societyUnit', 'feeConfig', 'assignedBy'])->get();
        return response()->json([
            'status' => 'success',
            'message' => 'Postbox allotments retrieved successfully',
            'data' => $pbAllotments
        ], 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'PB_ID' => 'required|exists:pb_units,PB_ID',
            'Status' => 'required|string|in:A,I',
            'ActiveDate' => ['required', 'regex:/^\d{2}\/\d{2}\/\d{4}$/', function ($attribute, $value, $fail) {
                try {
                    Carbon::createFromFormat('d/m/Y', $value);
                } catch (\Exception $e) {
                    $fail('The active date must be a valid date in dd/mm/yyyy format.');
                }
            }],
            'ValidDate' => ['nullable', 'regex:/^\d{2}\/\d{2}\/\d{4}$/', function ($attribute, $value, $fail) {
                if ($value) {
                    try {
                        Carbon::createFromFormat('d/m/Y', $value);
                    } catch (\Exception $e) {
                        $fail('The valid date must be a valid date in dd/mm/yyyy format.');
                    }
                }
            }],
            'SU_ID' => 'required|exists:soc_units,SU_ID',
            'Fees_Id' => 'required|exists:fees_config,FC_Id',
            'Asgn_By' => 'required|exists:users_master,U_Id',
        ], [
            'PB_ID.required' => 'The postbox unit ID is required.',
            'PB_ID.exists' => 'The selected postbox unit does not exist.',
            'Status.required' => 'The status is required.',
            'Status.in' => 'Status must be A (active) or I (inactive).',
            'ActiveDate.required' => 'The active date is required.',
            'SU_ID.required' => 'The society unit ID is required.',
            'SU_ID.exists' => 'The selected society unit does not exist.',
            'Fees_Id.required' => 'The fee configuration ID is required.',
            'Fees_Id.exists' => 'The selected fee configuration does not exist.',
            'Asgn_By.required' => 'The assigned by user ID is required.',
            'Asgn_By.exists' => 'The selected user does not exist.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $data = $validator->validated();
            $pbAllotment = PBAllotment::create($data);
            $pbAllotment->load(['pbUnit', 'societyUnit', 'feeConfig', 'assignedBy']);
            return response()->json([
                'status' => 'success',
                'message' => 'Postbox allotment created successfully',
                'data' => $pbAllotment
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while creating the postbox allotment: ' . $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        $pbAllotment = PBAllotment::with(['pbUnit', 'societyUnit', 'feeConfig', 'assignedBy'])->find($id);
        if (!$pbAllotment) {
            return response()->json([
                'status' => 'error',
                'message' => 'Postbox allotment not found'
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Postbox allotment retrieved successfully',
            'data' => $pbAllotment
        ], 200);
    }

    public function update(Request $request, $id)
    {
        $pbAllotment = PBAllotment::find($id);
        if (!$pbAllotment) {
            return response()->json([
                'status' => 'error',
                'message' => 'Postbox allotment not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'PB_ID' => 'required|exists:pb_units,PB_ID',
            'Status' => 'required|string|in:A,I',
            'ActiveDate' => ['required', 'regex:/^\d{2}\/\d{2}\/\d{4}$/', function ($attribute, $value, $fail) {
                try {
                    Carbon::createFromFormat('d/m/Y', $value);
                } catch (\Exception $e) {
                    $fail('The active date must be a valid date in dd/mm/yyyy format.');
                }
            }],
            'ValidDate' => ['nullable', 'regex:/^\d{2}\/\d{2}\/\d{4}$/', function ($attribute, $value, $fail) {
                if ($value) {
                    try {
                        Carbon::createFromFormat('d/m/Y', $value);
                    } catch (\Exception $e) {
                        $fail('The valid date must be a valid date in dd/mm/yyyy format.');
                    }
                }
            }],
            'SU_ID' => 'required|exists:soc_units,SU_ID',
            'Fees_Id' => 'required|exists:fees_config,FC_Id',
            'Asgn_By' => 'required|exists:users_master,U_Id',
        ], [
            'PB_ID.required' => 'The postbox unit ID is required.',
            'PB_ID.exists' => 'The selected postbox unit does not exist.',
            'Status.required' => 'The status is required.',
            'Status.in' => 'Status must be A (active) or I (inactive).',
            'ActiveDate.required' => 'The active date is required.',
            'SU_ID.required' => 'The society unit ID is required.',
            'SU_ID.exists' => 'The selected society unit does not exist.',
            'Fees_Id.required' => 'The fee configuration ID is required.',
            'Fees_Id.exists' => 'The selected fee configuration does not exist.',
            'Asgn_By.required' => 'The assigned by user ID is required.',
            'Asgn_By.exists' => 'The selected user does not exist.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $data = $validator->validated();
            $pbAllotment->update($data);
            $pbAllotment->load(['pbUnit', 'societyUnit', 'feeConfig', 'assignedBy']);
            return response()->json([
                'status' => 'success',
                'message' => 'Postbox allotment updated successfully',
                'data' => $pbAllotment
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while updating the postbox allotment.'
            ], 500);
        }
    }

    public function destroy($id)
    {
        $pbAllotment = PBAllotment::find($id);
        if (!$pbAllotment) {
            return response()->json([
                'status' => 'error',
                'message' => 'Postbox allotment not found'
            ], 404);
        }

        try {
            $pbAllotment->delete();
            return response()->json([
                'status' => 'success',
                'message' => 'Postbox allotment deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while deleting the postbox allotment.'
            ], 500);
        }
    }
}
