<?php

namespace App\Http\Controllers;

use App\Models\ModuleMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ModuleMasterController extends Controller
{
    public function index()
    {
        $modules = ModuleMaster::all();
        return response()->json([
            'status' => 'success',
            'message' => 'Modules retrieved successfully',
            'data' => $modules
        ], 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'ModuleName' => 'required|string|max:25',
            'Description' => 'required|string|max:25',
            'Status' => 'required|string|max:1|in:A,I',
        ], [
            'ModuleName.required' => 'Module name is required.',
            'ModuleName.max' => 'Module name must not exceed 25 characters.',
            'Description.required' => 'Description is required.',
            'Description.max' => 'Description must not exceed 25 characters.',
            'Status.required' => 'Status is required.',
            'Status.in' => 'Status must be A (Active) or I (Inactive).',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $module = ModuleMaster::create($validator->validated());
            return response()->json([
                'status' => 'success',
                'message' => 'Module created successfully',
                'data' => $module
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while creating the module.'
            ], 500);
        }
    }

    public function show($id)
    {
        $module = ModuleMaster::find($id);
        if (!$module) {
            return response()->json([
                'status' => 'error',
                'message' => 'Module not found'
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Module retrieved successfully',
            'data' => $module
        ], 200);
    }

    public function update(Request $request, $id)
    {
        $module = ModuleMaster::find($id);
        if (!$module) {
            return response()->json([
                'status' => 'error',
                'message' => 'Module not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'ModuleName' => 'sometimes|string|max:25',
            'Description' => 'sometimes|string|max:25',
            'Status' => 'sometimes|string|max:1|in:A,I',
        ], [
            'ModuleName.max' => 'Module name must not exceed 25 characters.',
            'Description.max' => 'Description must not exceed 25 characters.',
            'Status.in' => 'Status must be A (Active) or I (Inactive).',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $module->update($request->only(['ModuleName', 'Description', 'Status']));
            return response()->json([
                'status' => 'success',
                'message' => 'Module updated successfully',
                'data' => $module
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while updating the module.'
            ], 500);
        }
    }

    public function destroy($id)
    {
        $module = ModuleMaster::find($id);
        if (!$module) {
            return response()->json([
                'status' => 'error',
                'message' => 'Module not found'
            ], 404);
        }

        try {
            $module->delete();
            return response()->json([
                'status' => 'success',
                'message' => 'Module deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while deleting the module.'
            ], 500);
        }
    }
}
