<?php

namespace App\Http\Controllers;

use App\Models\MeetingInvitee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class MeetingInviteeController extends Controller
{
    /**
     * Get all MeetingInvitee records.
     */
    public function index()
    {
        $invitees = MeetingInvitee::with(['meetingBook', 'user'])->get();

        // Format dates to Indian format
        $invitees->each(function ($invitee) {
            $invitee->InviteSend = Carbon::parse($invitee->InviteSend)->format('d/m/Y H:i:s');
            $invitee->InviteRead = Carbon::parse($invitee->InviteRead)->format('d/m/Y H:i:s');
        });

        return response()->json($invitees);
    }

    /**
     * Get a specific MeetingInvitee record by MI_Id.
     */
    public function show($id)
    {
        $invitee = MeetingInvitee::with(['meetingBook', 'user'])->find($id);

        if (!$invitee) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        // Format dates to Indian format
        $invitee->InviteSend = Carbon::parse($invitee->InviteSend)->format('d/m/Y H:i:s');
        $invitee->InviteRead = Carbon::parse($invitee->InviteRead)->format('d/m/Y H:i:s');

        return response()->json($invitee);
    }

    /**
     * Create a new MeetingInvitee record.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'MB_Id'      => 'required|integer|exists:meeting_book,MB_Id',
            'U_Id'       => 'required|integer|exists:users_master,U_Id',
            'InviteSend' => 'required|date_format:d/m/Y H:i:s',
            'InviteRead' => 'nullable|date_format:d/m/Y H:i:s',
            'Attend'     => 'nullable|string|max:1', // 'Y', 'N', 'P'
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        try {
            // Convert dates to MySQL format
            $inviteSendDate = Carbon::createFromFormat('d/m/Y H:i:s', $request->input('InviteSend'))->format('Y-m-d H:i:s');
            $inviteReadDate = $request->input('InviteRead') ? Carbon::createFromFormat('d/m/Y H:i:s', $request->input('InviteRead'))->format('Y-m-d H:i:s') : null;
        } catch (\Exception $e) {
            return response()->json(['error' => 'Invalid date format. Use dd/mm/yyyy H:i:s'], 422);
        }

        $invitee = MeetingInvitee::create([
            'MB_Id'      => $request->input('MB_Id'),
            'U_Id'       => $request->input('U_Id'),
            'InviteSend' => $inviteSendDate,
            'InviteRead' => $inviteReadDate,
            'Attend'     => $request->input('Attend'),
        ]);

        return response()->json($invitee, 201);
    }

    /**
     * Update a MeetingInvitee record by MI_Id.
     */
    public function update(Request $request, $id)
    {
        $invitee = MeetingInvitee::find($id);

        if (!$invitee) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'MB_Id'      => 'required|integer|exists:meeting_book,MB_Id',
            'U_Id'       => 'required|integer|exists:users_master,U_Id',
            'InviteSend' => 'required|date_format:d/m/Y H:i:s',
            'InviteRead' => 'nullable|date_format:d/m/Y H:i:s',
            'Attend'     => 'nullable|string|max:1',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        try {
            // Convert dates to MySQL format
            $inviteSendDate = Carbon::createFromFormat('d/m/Y H:i:s', $request->input('InviteSend'))->format('Y-m-d H:i:s');
            $inviteReadDate = $request->input('InviteRead') ? Carbon::createFromFormat('d/m/Y H:i:s', $request->input('InviteRead'))->format('Y-m-d H:i:s') : null;
        } catch (\Exception $e) {
            return response()->json(['error' => 'Invalid date format. Use dd/mm/yyyy H:i:s'], 422);
        }

        $invitee->update([
            'MB_Id'      => $request->input('MB_Id'),
            'U_Id'       => $request->input('U_Id'),
            'InviteSend' => $inviteSendDate,
            'InviteRead' => $inviteReadDate,
            'Attend'     => $request->input('Attend'),
        ]);

        return response()->json($invitee);
    }

    /**
     * Delete a MeetingInvitee record by MI_Id.
     */
    public function destroy($id)
    {
        $invitee = MeetingInvitee::find($id);

        if (!$invitee) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        $invitee->delete();
        return response()->json(['message' => 'Deleted successfully']);
    }
}
