<?php

namespace App\Http\Controllers;

use App\Models\LockerType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class LockerTypeController extends Controller
{
    public function index()
    {
        $lockerTypes = LockerType::with('society')->get();
        return response()->json([
            'status' => 'success',
            'message' => 'Locker types retrieved successfully',
            'data' => $lockerTypes
        ], 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'Locker_Desc' => 'required|string|max:25',
            'SocId' => 'required|exists:society_master,SocId',
        ], [
            'Locker_Desc.required' => 'The locker description is required.',
            'Locker_Desc.string' => 'The locker description must be a string.',
            'Locker_Desc.max' => 'The locker description may not be greater than 25 characters.',
            'SocId.required' => 'The society ID is required.',
            'SocId.exists' => 'The selected society does not exist.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $data = $validator->validated();
            $lockerType = LockerType::create($data);
            $lockerType->load('society');
            return response()->json([
                'status' => 'success',
                'message' => 'Locker type created successfully',
                'data' => $lockerType
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while creating the locker type: ' . $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        $lockerType = LockerType::with('society')->find($id);
        if (!$lockerType) {
            return response()->json([
                'status' => 'error',
                'message' => 'Locker type not found'
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Locker type retrieved successfully',
            'data' => $lockerType
        ], 200);
    }

    public function update(Request $request, $id)
    {
        $lockerType = LockerType::find($id);
        if (!$lockerType) {
            return response()->json([
                'status' => 'error',
                'message' => 'Locker type not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'Locker_Desc' => 'required|string|max:25',
            'SocId' => 'required|exists:society_master,SocId',
        ], [
            'Locker_Desc.required' => 'The locker description is required.',
            'Locker_Desc.string' => 'The locker description must be a string.',
            'Locker_Desc.max' => 'The locker description may not be greater than 25 characters.',
            'SocId.required' => 'The society ID is required.',
            'SocId.exists' => 'The selected society does not exist.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $data = $validator->validated();
            $lockerType->update($data);
            $lockerType->load('society');
            return response()->json([
                'status' => 'success',
                'message' => 'Locker type updated successfully',
                'data' => $lockerType
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while updating the locker type.'
            ], 500);
        }
    }

    public function destroy($id)
    {
        $lockerType = LockerType::find($id);
        if (!$lockerType) {
            return response()->json([
                'status' => 'error',
                'message' => 'Locker type not found'
            ], 404);
        }

        try {
            $lockerType->delete();
            return response()->json([
                'status' => 'success',
                'message' => 'Locker type deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while deleting the locker type.'
            ], 500);
        }
    }
}
