<?php

namespace App\Http\Controllers;

use App\Models\GuardDutyRegister;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class GuardDutyRegisterController extends Controller
{
    /**
     * Get all GuardDutyRegister records.
     */
    public function index()
    {
        $dutyRegisters = GuardDutyRegister::with(['guardRegistration', 'dutyPoint'])->get();
        return response()->json($dutyRegisters);
    }

    /**
     * Get a specific GuardDutyRegister record by GD_Id.
     */
    public function show($id)
    {
        $dutyRegister = GuardDutyRegister::with(['guardRegistration', 'dutyPoint'])->find($id);

        if (!$dutyRegister) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        return response()->json($dutyRegister);
    }

    /**
     * Create a new GuardDutyRegister record.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'GR_Id' => 'required|integer|exists:guard_registration,GR_Id',
            'DutyOn' => 'nullable|date_format:d/m/Y H:i:s',
            'DutyOff' => 'nullable|date_format:d/m/Y H:i:s|after:DutyOn',
            'DP_Id' => 'required|integer|exists:duty_points,DP_Id',
            'Status' => 'nullable|string|in:A,I|max:1',
            'DutyType' => 'nullable|string|in:F,P|max:1',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        DB::beginTransaction(); // Start transaction

        try {
            $validated = $validator->validated();

            // Convert date fields to MySQL format
            $validated['DutyOn'] = $request->filled('DutyOn')
                ? Carbon::createFromFormat('d/m/Y H:i:s', $request->input('DutyOn'), 'Asia/Kolkata')
                    ->format('Y-m-d H:i:s')
                : null;
            $validated['DutyOff'] = $request->filled('DutyOff')
                ? Carbon::createFromFormat('d/m/Y H:i:s', $request->input('DutyOff'), 'Asia/Kolkata')
                    ->format('Y-m-d H:i:s')
                : null;

            // Create Guard Duty Register
            $dutyRegister = GuardDutyRegister::create($validated);

            DB::commit(); // Commit transaction

            // Eager load the related guard registration and duty point
            $dutyRegister->load('guardRegistration', 'dutyPoint');

            return response()->json([
                'message' => 'Guard duty registered successfully!',
                'duty' => $dutyRegister
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack(); // Rollback on failure
            return response()->json(['error' => 'Guard duty registration failed!', 'details' => $e->getMessage()], 500);
        }
    }

    /**
     * Update a GuardDutyRegister record by GD_Id.
     */
    public function update(Request $request, $id)
    {
        $dutyRegister = GuardDutyRegister::find($id);

        if (!$dutyRegister) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'GR_Id' => 'nullable|integer|exists:guard_registration,GR_Id',
            'DutyOn' => 'nullable|date_format:d/m/Y H:i:s',
            'DutyOff' => 'nullable|date_format:d/m/Y H:i:s|after:DutyOn',
            'DP_Id' => 'nullable|integer|exists:duty_points,DP_Id',
            'Status' => 'nullable|string|in:A,I|max:1',
            'DutyType' => 'nullable|string|in:F,P|max:1',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $validated = array_filter($validator->validated());

        // Convert date fields to MySQL format
        if ($request->filled('DutyOn')) {
            $validated['DutyOn'] = Carbon::createFromFormat('d/m/Y H:i:s', $request->input('DutyOn'), 'Asia/Kolkata')
                ->format('Y-m-d H:i:s');
        }
        if ($request->filled('DutyOff')) {
            $validated['DutyOff'] = Carbon::createFromFormat('d/m/Y H:i:s', $request->input('DutyOff'), 'Asia/Kolkata')
                ->format('Y-m-d H:i:s');
        }

        $dutyRegister->update($validated);

        $dutyRegister->load('guardRegistration', 'dutyPoint');

        return response()->json($dutyRegister);
    }

    /**
     * Delete a GuardDutyRegister record by GD_Id.
     */
    public function destroy($id)
    {
        $dutyRegister = GuardDutyRegister::find($id);

        if (!$dutyRegister) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        $dutyRegister->delete();
        return response()->json(['message' => 'Deleted successfully']);
    }
}
