<?php

namespace App\Http\Controllers;

use App\Models\FreeLRating;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class FreeLRatingController extends Controller
{
    /**
     * Get all FreeLRating records.
     */
    public function index()
    {
        $ratings = FreeLRating::with(['freelancerRegister', 'user'])->get();
        return response()->json($ratings);
    }

    /**
     * Get a specific FreeLRating record by FLR_Id.
     */
    public function show($id)
    {
        $rating = FreeLRating::with(['freelancerRegister', 'user'])->find($id);

        if (!$rating) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        return response()->json($rating);
    }

    /**
     * Create a new FreeLRating record.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'FL_Id' => 'required|integer|exists:freelancer_register,FL_Id',
            'U_Id' => 'required|integer|exists:users_master,U_Id',
            'Status' => 'required|integer|between:1,5',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        DB::beginTransaction(); // Start transaction

        try {
            $validated = $validator->validated();

            // Create FreeLRating record
            $rating = FreeLRating::create($validated);

            DB::commit(); // Commit transaction

            // Eager load the related data
            $rating->load('freelancerRegister', 'user');

            return response()->json([
                'message' => 'Freelancer rating registered successfully!',
                'rating' => $rating
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack(); // Rollback on failure
            return response()->json(['error' => 'Freelancer rating registration failed!', 'details' => $e->getMessage()], 500);
        }
    }

    /**
     * Update a FreeLRating record by FLR_Id.
     */
    public function update(Request $request, $id)
    {
        $rating = FreeLRating::find($id);

        if (!$rating) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'FL_Id' => 'nullable|integer|exists:freelancer_register,FL_Id',
            'U_Id' => 'nullable|integer|exists:users_master,U_Id',
            'Status' => 'nullable|integer|between:1,5',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $validated = array_filter($validator->validated());

        $rating->update($validated);

        $rating->load('freelancerRegister', 'user');

        return response()->json($rating);
    }

    /**
     * Delete a FreeLRating record by FLR_Id.
     */
    public function destroy($id)
    {
        $rating = FreeLRating::find($id);

        if (!$rating) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        $rating->delete();
        return response()->json(['message' => 'Deleted successfully']);
    }
}
