<?php

namespace App\Http\Controllers;

use App\Models\FreeLPackMast;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class FreeLPackMastController extends Controller
{
    /**
     * Get all FreeLPackMast records.
     */
    public function index()
    {
        $packages = FreeLPackMast::with('society')->get();
        return response()->json($packages);
    }

    /**
     * Get a specific FreeLPackMast record by FLP_Id.
     */
    public function show($id)
    {
        $package = FreeLPackMast::with('society')->find($id);

        if (!$package) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        return response()->json($package);
    }

    /**
     * Create a new FreeLPackMast record.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'FLP_Desc' => 'required|string|max:25',
            'FLP_Qty' => 'required|numeric|min:0',
            'FLP_Rate' => 'required|numeric|min:0',
            'SocId' => 'required|integer|exists:society_master,SocId',
            'FLP_Type' => 'nullable|string|in:S,P|max:1',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        DB::beginTransaction(); // Start transaction

        try {
            $validated = $validator->validated();

            // Create FreeLPackMast record
            $package = FreeLPackMast::create($validated);

            DB::commit(); // Commit transaction

            // Eager load the related society
            $package->load('society');

            return response()->json([
                'message' => 'Freelancer package registered successfully!',
                'package' => $package
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack(); // Rollback on failure
            return response()->json(['error' => 'Freelancer package registration failed!', 'details' => $e->getMessage()], 500);
        }
    }

    /**
     * Update a FreeLPackMast record by FLP_Id.
     */
    public function update(Request $request, $id)
    {
        $package = FreeLPackMast::find($id);

        if (!$package) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'FLP_Desc' => 'nullable|string|max:25',
            'FLP_Qty' => 'nullable|numeric|min:0',
            'FLP_Rate' => 'nullable|numeric|min:0',
            'SocId' => 'nullable|integer|exists:society_master,SocId',
            'FLP_Type' => 'nullable|string|in:S,P|max:1',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $validated = array_filter($validator->validated());

        $package->update($validated);

        $package->load('society');

        return response()->json($package);
    }

    /**
     * Delete a FreeLPackMast record by FLP_Id.
     */
    public function destroy($id)
    {
        $package = FreeLPackMast::find($id);

        if (!$package) {
            return response()->json(['message' => 'Record not found'], 404);
        }

        $package->delete();
        return response()->json(['message' => 'Deleted successfully']);
    }
}
