<?php

namespace App\Http\Controllers;

use App\Models\Fee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class FeeController extends Controller
{
    public function index()
    {
        $fees = Fee::all();
        return response()->json([
            'status' => 'success',
            'message' => 'Fees retrieved successfully',
            'data' => $fees
        ], 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'Fs_Desc' => 'required|string|max:100',
            'Status' => 'required|string|in:A,I',
        ], [
            'Fs_Desc.required' => 'The fee description is required.',
            'Fs_Desc.string' => 'The fee description must be a string.',
            'Fs_Desc.max' => 'The fee description may not be greater than 100 characters.',
            'Status.required' => 'The status is required.',
            'Status.in' => 'The status must be A (active) or I (inactive).',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $data = $validator->validated();
            $fee = Fee::create($data);
            return response()->json([
                'status' => 'success',
                'message' => 'Fee created successfully',
                'data' => $fee
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while creating the fee: ' . $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        $fee = Fee::find($id);
        if (!$fee) {
            return response()->json([
                'status' => 'error',
                'message' => 'Fee not found'
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Fee retrieved successfully',
            'data' => $fee
        ], 200);
    }

    public function update(Request $request, $id)
    {
        $fee = Fee::find($id);
        if (!$fee) {
            return response()->json([
                'status' => 'error',
                'message' => 'Fee not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'Fs_Desc' => 'required|string|max:100',
            'Status' => 'required|string|in:A,I',
        ], [
            'Fs_Desc.required' => 'The fee description is required.',
            'Fs_Desc.string' => 'The fee description must be a string.',
            'Fs_Desc.max' => 'The fee description may not be greater than 100 characters.',
            'Status.required' => 'The status is required.',
            'Status.in' => 'The status must be A (active) or I (inactive).',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $data = $validator->validated();
            $fee->update($data);
            return response()->json([
                'status' => 'success',
                'message' => 'Fee updated successfully',
                'data' => $fee
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while updating the fee.'
            ], 500);
        }
    }

    public function destroy($id)
    {
        $fee = Fee::find($id);
        if (!$fee) {
            return response()->json([
                'status' => 'error',
                'message' => 'Fee not found'
            ], 404);
        }

        try {
            $fee->delete();
            return response()->json([
                'status' => 'success',
                'message' => 'Fee deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while deleting the fee.'
            ], 500);
        }
    }
}
