<?php

namespace App\Http\Controllers;

use App\Models\ComDesi;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ComDesiController extends Controller
{
    public function index()
    {
        $comDesis = ComDesi::all();
        return response()->json([
            'status' => 'success',
            'message' => 'Community designations retrieved successfully',
            'data' => $comDesis
        ], 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'MemDesi_Name' => 'nullable|string|max:50|unique:com_desi,MemDesi_Name',
            'SericeAreas' => 'nullable|string|max:50',
        ], [
            'MemDesi_Name.max' => 'Designation name must not exceed 50 characters.',
            'MemDesi_Name.unique' => 'This designation name already exists.',
            'SericeAreas.max' => 'Service areas must not exceed 50 characters.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $comDesi = ComDesi::create($validator->validated());
            return response()->json([
                'status' => 'success',
                'message' => 'Community designation created successfully',
                'data' => $comDesi
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while creating the community designation.'
            ], 500);
        }
    }

    public function show($id)
    {
        $comDesi = ComDesi::find($id);
        if (!$comDesi) {
            return response()->json([
                'status' => 'error',
                'message' => 'Community designation not found'
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Community designation retrieved successfully',
            'data' => $comDesi
        ], 200);
    }

    public function update(Request $request, $id)
    {
        $comDesi = ComDesi::find($id);
        if (!$comDesi) {
            return response()->json([
                'status' => 'error',
                'message' => 'Community designation not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'MemDesi_Name' => 'nullable|string|max:50|unique:com_desi,MemDesi_Name,' . $id . ',MemDesi_ID',
            'SericeAreas' => 'nullable|string|max:50',
        ], [
            'MemDesi_Name.max' => 'Designation name must not exceed 50 characters.',
            'MemDesi_Name.unique' => 'This designation name already exists.',
            'SericeAreas.max' => 'Service areas must not exceed 50 characters.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $comDesi->update($request->only(['MemDesi_Name', 'SericeAreas']));
            return response()->json([
                'status' => 'success',
                'message' => 'Community designation updated successfully',
                'data' => $comDesi
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while updating the community designation.'
            ], 500);
        }
    }

    public function destroy($id)
    {
        $comDesi = ComDesi::find($id);
        if (!$comDesi) {
            return response()->json([
                'status' => 'error',
                'message' => 'Community designation not found'
            ], 404);
        }

        try {
            $comDesi->delete();
            return response()->json([
                'status' => 'error',
                'message' => 'Community designation deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An unexpected error occurred while deleting the community designation.'
            ], 500);
        }
    }
}
