<?php

namespace App\Http\Controllers;

use App\Models\CoMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class CoMasterController extends Controller
{
    public function index()
    {
        $coMasters = CoMaster::with(['branches', 'ledgerGroups', 'parties'])->get();

        // Calculate financial period for each CoMaster and format response
        $coMastersWithFinancialPeriod = $coMasters->map(function ($coMaster) {
            $financialPeriod = null;
            if ($coMaster->CoFin) {
                $today = Carbon::today();
                $currentMonth = $today->month;
                $currentYear = $today->year;

                $finStartMonth = $coMaster->CoFin;
                if ($currentMonth >= $finStartMonth) {
                    $startYear = $currentYear;
                } else {
                    $startYear = $currentYear - 1;
                }

                $startDate = Carbon::create($startYear, $finStartMonth, 1)->startOfDay();
                $endDate = $startDate->copy()->addYear()->subDay();

                $financialPeriod = [
                    'start_date' => $startDate->format('d-m-Y'),
                    'end_date' => $endDate->format('d-m-Y'),
                ];
            }

            // Merge coMaster attributes with financialPeriod
            return array_merge($coMaster->toArray(), ['financialPeriod' => $financialPeriod]);
        });

        return response()->json($coMastersWithFinancialPeriod, 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'co_grp' => 'required|string|max:5',
            'co_nam' => 'required|string|max:40',
            'co_ad1' => 'nullable|string|max:40',
            'co_ad2' => 'nullable|string|max:40',
            'co_cty' => 'nullable|string|max:33',
            'co_pin' => 'nullable|string|max:7',
            'co_phn' => 'nullable|string|max:40',
            'co_eml' => 'nullable|email|max:40',
            'co_url' => 'nullable|url|max:40',
            'co_pan' => 'nullable|string|max:20',
            'co_tds' => 'nullable|string|max:20',
            'co_pac' => 'nullable|string|max:20',
            'co_tdc' => 'nullable|string|max:20',
            'co_lic' => 'nullable|string|max:20',
            'co_reg' => 'nullable|string|max:20',
            'co_st_n' => 'nullable|string|max:40',
            'co_cst' => 'nullable|string|max:40',
            'co_vat' => 'nullable|string|max:40',
            'co_ecc' => 'nullable|string|max:40',
            'co_exc' => 'nullable|string|max:20',
            'co_rng' => 'nullable|string|max:20',
            'tr_mod' => 'nullable|string|max:1',
            'co_bak' => 'nullable|numeric',
            'co_ent' => 'nullable|string|max:1',
            'co_typ' => 'nullable|string|max:1',
            'CoFin' => 'nullable|integer|min:1|max:12',
            'lst_bk' => 'nullable|date',
            'bs_typ' => 'nullable|string|max:1',
            'co_qty' => 'nullable|numeric',
            'co_pss' => 'nullable|string|max:7',
            'op_id' => 'nullable|integer',
            'dt_tm' => 'nullable|string|max:10',
            'logo' => 'nullable|string|max:200',
            'dept_id' => 'nullable|string|max:50',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $coMaster = CoMaster::create($request->all());

        // Calculate financial period based on CoFin
        $financialPeriod = null;
        if ($request->has('CoFin') && $request->CoFin) {
            $today = Carbon::today();
            $currentMonth = $today->month;
            $currentYear = $today->year;

            $finStartMonth = $request->CoFin;
            if ($currentMonth >= $finStartMonth) {
                $startYear = $currentYear;
            } else {
                $startYear = $currentYear - 1;
            }

            $startDate = Carbon::create($startYear, $finStartMonth, 1)->startOfDay();
            $endDate = $startDate->copy()->addYear()->subDay();

            $financialPeriod = [
                'start_date' => $startDate->format('d-m-Y'),
                'end_date' => $endDate->format('d-m-Y'),
            ];
        }

        return response()->json([
            'coMaster' => $coMaster->load(['branches', 'ledgerGroups', 'parties']),
            'financialPeriod' => $financialPeriod,
        ], 201);
    }

    public function show($id)
    {
        $coMaster = CoMaster::with(['branches', 'ledgerGroups', 'parties'])->findOrFail($id);

        // Calculate financial period based on CoFin
        $financialPeriod = null;
        if ($coMaster->CoFin) {
            $today = Carbon::today();
            $currentMonth = $today->month;
            $currentYear = $today->year;

            $finStartMonth = $coMaster->CoFin;
            if ($currentMonth >= $finStartMonth) {
                $startYear = $currentYear;
            } else {
                $startYear = $currentYear - 1;
            }

            $startDate = Carbon::create($startYear, $finStartMonth, 1)->startOfDay();
            $endDate = $startDate->copy()->addYear()->subDay();

            $financialPeriod = [
                'start_date' => $startDate->format('d-m-Y'),
                'end_date' => $endDate->format('d-m-Y'),
            ];
        }

        // Merge coMaster attributes with financialPeriod
        return response()->json(array_merge($coMaster->toArray(), ['financialPeriod' => $financialPeriod]), 200);
    }

    public function update(Request $request, $id)
    {
        $coMaster = CoMaster::findOrFail($id);
        $validator = Validator::make($request->all(), [
            'co_grp' => 'required|string|max:5',
            'co_nam' => 'required|string|max:40',
            'co_ad1' => 'nullable|string|max:40',
            'co_ad2' => 'nullable|string|max:40',
            'co_cty' => 'nullable|string|max:33',
            'co_pin' => 'nullable|string|max:7',
            'co_phn' => 'nullable|string|max:40',
            'co_eml' => 'nullable|email|max:40',
            'co_url' => 'nullable|url|max:40',
            'co_pan' => 'nullable|string|max:20',
            'co_tds' => 'nullable|string|max:20',
            'co_pac' => 'nullable|string|max:20',
            'co_tdc' => 'nullable|string|max:20',
            'co_lic' => 'nullable|string|max:20',
            'co_reg' => 'nullable|string|max:20',
            'co_st_n' => 'nullable|string|max:40',
            'co_cst' => 'nullable|string|max:40',
            'co_vat' => 'nullable|string|max:40',
            'co_ecc' => 'nullable|string|max:40',
            'co_exc' => 'nullable|string|max:20',
            'co_rng' => 'nullable|string|max:20',
            'tr_mod' => 'nullable|string|max:1',
            'co_bak' => 'nullable|numeric',
            'co_ent' => 'nullable|string|max:1',
            'co_typ' => 'nullable|string|max:1',
            'CoFin' => 'nullable|integer|min:1|max:12',
            'lst_bk' => 'nullable|date',
            'bs_typ' => 'nullable|string|max:1',
            'co_qty' => 'nullable|numeric',
            'co_pss' => 'nullable|string|max:7',
            'op_id' => 'nullable|integer',
            'dt_tm' => 'nullable|string|max:10',
            'logo' => 'nullable|string|max:200',
            'dept_id' => 'nullable|string|max:50',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $coMaster->update($request->all());
        return response()->json($coMaster->load(['branches', 'ledgerGroups', 'parties']), 200);
    }

    public function destroy($id)
    {
        $coMaster = CoMaster::findOrFail($id);
        $coMaster->delete();
        return response()->json(null, 204);
    }
}
