<?php

namespace App\Http\Controllers;

use App\Models\AmenityConfig;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class AmenityConfigController extends Controller
{
    /**
     * Get all AmenityConfig records.
     */
    public function index()
    {
        $configs = AmenityConfig::with(['amenity', 'society', 'feesConfig'])->get();
        return response()->json($configs);
    }

    /**
     * Get a specific AmenityConfig record by AC_ID.
     */
    public function show($id)
    {
        $config = AmenityConfig::with(['amenity', 'society', 'feesConfig'])->find($id);

        if (!$config) {
            return response()->json(['message' => 'Amenity config not found'], 404);
        }

        return response()->json($config);
    }

    /**
     * Create a new AmenityConfig record.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'AM_ID' => 'required|integer|exists:amenity_masters,AM_ID',
            'SocId' => 'required|integer|exists:society_master,SocId',
            'FC_Id' => 'required|integer|exists:fees_config,FC_Id',
            'Name' => 'required|string|max:25',
            'MaxCapacity' => 'required|integer|min:1',
            'Status' => 'required|string|in:A,I',
            'TimeSlot' => 'required|integer|min:1',
            'FeesUnit' => 'required|integer|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $config = AmenityConfig::create([
            'AM_ID' => $request->input('AM_ID'),
            'SocId' => $request->input('SocId'),
            'FC_Id' => $request->input('FC_Id'),
            'Name' => $request->input('Name'),
            'MaxCapacity' => $request->input('MaxCapacity'),
            'Status' => $request->input('Status'),
            'TimeSlot' => $request->input('TimeSlot'),
            'FeesUnit' => $request->input('FeesUnit'),
        ]);

        return response()->json($config, 201);
    }

    /**
     * Update an AmenityConfig record by AC_ID.
     */
    public function update(Request $request, $id)
    {
        $config = AmenityConfig::find($id);

        if (!$config) {
            return response()->json(['message' => 'Amenity config not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'AM_ID' => 'nullable|integer|exists:amenity_masters,AM_ID',
            'SocId' => 'nullable|integer|exists:society_master,SocId',
            'FC_Id' => 'nullable|integer|exists:fees_config,FC_Id',
            'Name' => 'nullable|string|max:25',
            'MaxCapacity' => 'nullable|integer|min:1',
            'Status' => 'nullable|string|in:A,I',
            'TimeSlot' => 'nullable|integer|min:1',
            'FeesUnit' => 'nullable|integer|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $updateData = array_filter($request->only(['AM_ID', 'SocId', 'FC_Id', 'Name', 'MaxCapacity', 'Status', 'TimeSlot', 'FeesUnit']));
        $config->update($updateData);

        return response()->json($config);
    }

    /**
     * Delete an AmenityConfig record by AC_ID.
     */
    public function delete($id)
    {
        $config = AmenityConfig::find($id);

        if (!$config) {
            return response()->json(['message' => 'Amenity config not found'], 404);
        }

        $config->delete();
        return response()->json(['message' => 'Amenity config deleted successfully']);
    }
}
