<?php

namespace App\Http\Controllers;

use App\Models\AmenityBooking;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class AmenityBookingController extends Controller
{
    /**
     * Get all AmenityBooking records.
     */
    public function index()
    {
        $bookings = AmenityBooking::with(['amenityConfig', 'user', 'approver'])->get();
        return response()->json($bookings);
    }

    /**
     * Get a specific AmenityBooking record by AB_ID.
     */
    public function show($id)
    {
        $booking = AmenityBooking::with(['amenityConfig', 'user', 'approver'])->find($id);

        if (!$booking) {
            return response()->json(['message' => 'Amenity booking not found'], 404);
        }

        return response()->json($booking);
    }

    /**
     * Create a new AmenityBooking record.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'AC_ID' => 'required|integer|exists:amenity_configs,AC_ID',
            'U_Id' => 'required|integer|exists:users_master,U_Id',
            'StartDateTime' => 'nullable|date_format:d/m/Y H:i:s',
            'EndDateTime' => 'nullable|date_format:d/m/Y H:i:s|after:StartDateTime',
            'QtyRequested' => 'nullable|integer|min:1',
            'QtyBooked' => 'nullable|integer|min:0',
            'Amount' => 'nullable|numeric|min:0',
            'EB_Disc' => 'nullable|numeric|min:0',
            'Blk_Disc' => 'nullable|numeric|min:0',
            'Spl_Cat_Disc' => 'nullable|numeric|min:0',
            'Late_Chg' => 'nullable|numeric|min:0',
            'Int_Chg' => 'nullable|numeric|min:0',
            'Req_Date' => 'nullable|date_format:d/m/Y H:i:s',
            'Approved_By' => 'nullable|integer|exists:users_master,U_Id',
            'Approved_Date' => 'nullable|date_format:d/m/Y H:i:s',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $validated = $validator->validated();

        // Convert date fields to MySQL format
        $validated['StartDateTime'] = $request->filled('StartDateTime')
            ? Carbon::createFromFormat('d/m/Y H:i:s', $request->input('StartDateTime'), 'Asia/Kolkata')
                ->format('Y-m-d H:i:s')
            : null;
        $validated['EndDateTime'] = $request->filled('EndDateTime')
            ? Carbon::createFromFormat('d/m/Y H:i:s', $request->input('EndDateTime'), 'Asia/Kolkata')
                ->format('Y-m-d H:i:s')
            : null;
        $validated['Req_Date'] = $request->filled('Req_Date')
            ? Carbon::createFromFormat('d/m/Y H:i:s', $request->input('Req_Date'), 'Asia/Kolkata')
                ->format('Y-m-d H:i:s')
            : null;
        $validated['Approved_Date'] = $request->filled('Approved_Date')
            ? Carbon::createFromFormat('d/m/Y H:i:s', $request->input('Approved_Date'), 'Asia/Kolkata')
                ->format('Y-m-d H:i:s')
            : null;

        $booking = AmenityBooking::create($validated);

        return response()->json($booking, 201);
    }

    /**
     * Update an AmenityBooking record by AB_ID.
     */
    public function update(Request $request, $id)
    {
        $booking = AmenityBooking::find($id);

        if (!$booking) {
            return response()->json(['message' => 'Amenity booking not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'AC_ID' => 'nullable|integer|exists:amenity_configs,AC_ID',
            'U_Id' => 'nullable|integer|exists:users_master,U_Id',
            'StartDateTime' => 'nullable|date_format:d/m/Y H:i:s',
            'EndDateTime' => 'nullable|date_format:d/m/Y H:i:s|after:StartDateTime',
            'QtyRequested' => 'nullable|integer|min:1',
            'QtyBooked' => 'nullable|integer|min:0',
            'Amount' => 'nullable|numeric|min:0',
            'EB_Disc' => 'nullable|numeric|min:0',
            'Blk_Disc' => 'nullable|numeric|min:0',
            'Spl_Cat_Disc' => 'nullable|numeric|min:0',
            'Late_Chg' => 'nullable|numeric|min:0',
            'Int_Chg' => 'nullable|numeric|min:0',
            'Req_Date' => 'nullable|date_format:d/m/Y H:i:s',
            'Approved_By' => 'nullable|integer|exists:users_master,U_Id',
            'Approved_Date' => 'nullable|date_format:d/m/Y H:i:s',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $validated = array_filter($validator->validated());

        // Convert date fields to MySQL format
        if ($request->filled('StartDateTime')) {
            $validated['StartDateTime'] = Carbon::createFromFormat('d/m/Y H:i:s', $request->input('StartDateTime'), 'Asia/Kolkata')
                ->format('Y-m-d H:i:s');
        }
        if ($request->filled('EndDateTime')) {
            $validated['EndDateTime'] = Carbon::createFromFormat('d/m/Y H:i:s', $request->input('EndDateTime'), 'Asia/Kolkata')
                ->format('Y-m-d H:i:s');
        }
        if ($request->filled('Req_Date')) {
            $validated['Req_Date'] = Carbon::createFromFormat('d/m/Y H:i:s', $request->input('Req_Date'), 'Asia/Kolkata')
                ->format('Y-m-d H:i:s');
        }
        if ($request->filled('Approved_Date')) {
            $validated['Approved_Date'] = Carbon::createFromFormat('d/m/Y H:i:s', $request->input('Approved_Date'), 'Asia/Kolkata')
                ->format('Y-m-d H:i:s');
        }

        $booking->update($validated);

        return response()->json($booking);
    }

    /**
     * Delete an AmenityBooking record by AB_ID.
     */
    public function destroy($id)
    {
        $booking = AmenityBooking::find($id);

        if (!$booking) {
            return response()->json(['message' => 'Amenity booking not found'], 404);
        }

        $booking->delete();
        return response()->json(['message' => 'Amenity booking deleted successfully']);
    }
}

